package fr.test.ooo;

import java.awt.Color;
import java.net.URL;

import javax.swing.ImageIcon;

import ooo.connector.BootstrapSocketConnector;

import com.sun.star.awt.FontSlant;
import com.sun.star.awt.FontWeight;
import com.sun.star.awt.Point;
import com.sun.star.awt.Size;
import com.sun.star.beans.Property;
import com.sun.star.beans.PropertyValue;
import com.sun.star.beans.UnknownPropertyException;
import com.sun.star.beans.XPropertySet;
import com.sun.star.bridge.XUnoUrlResolver;
import com.sun.star.comp.helper.BootstrapException;
import com.sun.star.drawing.XShape;
import com.sun.star.drawing.XShapes;
import com.sun.star.frame.XComponentLoader;
import com.sun.star.frame.XFrame;
import com.sun.star.frame.XModel;
import com.sun.star.frame.XStorable;
import com.sun.star.io.IOException;
import com.sun.star.lang.IllegalArgumentException;
import com.sun.star.lang.IndexOutOfBoundsException;
import com.sun.star.lang.WrappedTargetException;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XMultiComponentFactory;
import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.sheet.XSpreadsheet;
import com.sun.star.table.XCell;
import com.sun.star.text.TextContentAnchorType;
import com.sun.star.text.WrapTextMode;
import com.sun.star.text.XText;
import com.sun.star.text.XTextCursor;
import com.sun.star.text.XTextTable;
import com.sun.star.uno.Exception;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XComponentContext;
import com.sun.star.util.CloseVetoException;
import com.sun.star.util.XCloseable;

import fr.test.utils.DocumentType;
import fr.test.utils.SaveFormat;
import fr.test.utils.Utils;

/**
 * Classe de méthodes utilitaires pour OOo
 * 
 * @author Sylvain Saurel
 *
 */
public class OOoUtils {
	
	/** Chemin vers le répertoire des programmes de OOo */
	public static String OOO_PROGRAM_PATH = "C:\\Program Files\\OpenOffice.org 3\\program";
	
	/**
	 * Constructeur privé car classe utilisée
	 * seulement en contexte statique
	 * 
	 */
	private OOoUtils(){
	}
	
	/**
	 * Permet d'afficher la liste détaillée des propriétés
	 * présentes dans un XPropertySet
	 * 
	 * @param title Le nom de l'objet auquel est rattaché le
	 * XPropertySet
	 * @param pSet L'objet XPropertySet dont on souhaite connaître le contenu
	 */
    public static void showProperties(String title, XPropertySet pSet) {
    	if(title != null  &&  pSet != null){
    		System.out.println("\n" + title + "\n");
    		Property[] props = pSet.getPropertySetInfo().getProperties();
    		
    		if(props != null){
    			int length = props.length;
    			for(int i = 0; i < length; i++) {
    				Property p = props[i];
    				String value = "<null>";
    				Object o = null;
					try {
						o = (Object) pSet.getPropertyValue(p.Name);
					} catch (UnknownPropertyException upe) {
						System.out.println("Propriété inconnue : " + upe.getMessage());
					} catch (WrappedTargetException wte) {
						System.out.println("Erreur à la récupération de la propriété : " + wte.getMessage());
					}
					
    				if(o != null){
    					value = o.toString();
    				}
    				
    				System.out.println("   Name = " + p.Name + ", Type = " + p.Type + ", Value = " + value);
    			}
    		}    		
    	}
    }
    
    /**
     * Méthode permettant de récupérer une interface définie en entrée
     * à partir d'un objet passé en paramètre
     * 
     * @param <T> Type de l'interface que l'on souhaite obtenir
     * @param clazz Classe de l'interface que l'on souhaite obtenir
     * @param obj Objet dont on souhaite récupérer l'implémentation
     * de l'interface <T>
     * @return L'implémentation de l'interface <T> pour obj, <code>null</code>
     * sinon
     */
    @SuppressWarnings("unchecked")
	public static <T> T getInterface(Class<T> clazz, Object obj){
    	return (T)UnoRuntime.queryInterface(clazz, obj);
    }
    
    /**
     * Récupération d'une instance locale de XComponentLoader
     * 
     * @return L'instance locale de XComponentLoader, <code>null</code> en 
     * cas d'erreur
     * @throws BootstrapException Exception lancée à la récupération du bootstrap
     * @throws Exception Exception lancée à la récupération du desktop
     */
    public static XComponentLoader getLocalXComponentLoader() throws BootstrapException, Exception{
    	// Récupération d'un contexte en précisant le chemin vers les exécutables OOo
    	XComponentContext xContext = BootstrapSocketConnector.bootstrap(OOO_PROGRAM_PATH);
    	// Accès au gestionnaire de services
        XMultiComponentFactory xmulticomponentfactory = xContext.getServiceManager();
        // Création du service XDesktop
        Object xDesktop = xmulticomponentfactory.createInstanceWithContext("com.sun.star.frame.Desktop", xContext);
        // Récupération d'une implémentation de l'interface XComponentLoader
        return OOoUtils.getInterface(XComponentLoader.class, xDesktop);
    }
    
    /**
     * Récupération d'une instance distante de XComponentLoader
     * 
     * @param ip IP de la machine distante 
     * @param port Port de la machine distante
     * @return L'instance distante de XComponentLoader, <code>null</code> en cas d'ereur
     * @throws BootstrapException Exception lancée à la récupération du bootstrap
     * @throws Exception Exception lancée à la récupération du desktop
     */
    public static XComponentLoader getRemoteXComponentLoader(String ip, String port) throws BootstrapException, Exception{
    	XComponentLoader xComponentLoader = null;
    	
    	if(ip != null  &&  port != null){
    		xComponentLoader = getRemoteXComponentLoader(new StringBuilder("uno:socket,host=").
    				append(ip).append(",port=").append(port).append(";urp;StarOffice.ServiceManager").
    				toString());
    	}
    	
    	return xComponentLoader;
    }
	
    /**
     * Récupération d'une instance distante de XComponentLoader
     * 
     * @param url Url vers la machine distante sur laquelle on souhaite récupérer
     * le XComponentLoader
     * NB : Url de la forme uno:socket,host=<IP>,port=<Port>;urp;StarOffice.ServiceManager
     * @return L'instance distante de XComponentLoader, <code>null</code> en cas d'erreur
     * @throws BootstrapException Exception lancée à la récupération du bootstrap
     * @throws Exception Exception lancée à la récupération du desktop
     */
    public static XComponentLoader getRemoteXComponentLoader(String url) throws BootstrapException, Exception{
    	XComponentLoader xComponentLoader = null;
    	
    	if(url != null){
    		// Récupération d'un contexte OOo
    		//XComponentContext xcomponentcontext = Bootstrap.createInitialComponentContext(null);
    		XComponentContext xComponentContext = BootstrapSocketConnector.bootstrap(OOO_PROGRAM_PATH);
    		// Récupération du gestionnaire de service associé à ce contexte
    		XMultiComponentFactory xMultiComponentFactory = xComponentContext.getServiceManager();
    		// Création du service d'accès distant
    		Object objectUrlResolver = xMultiComponentFactory.createInstanceWithContext("com.sun.star.bridge.UnoUrlResolver",
    				xComponentContext);
    		// Récupération d'une implémentation de l'interface XUnoUrlResolver
    		XUnoUrlResolver xUrlResolver = OOoUtils.getInterface(XUnoUrlResolver.class, objectUrlResolver);
    		// Récupération d'une instance OOo sur le poste distant (processus OOo déjà lancé, ou création d'un nouveau)
    		Object objectInitial = xUrlResolver.resolve(url);
    		// Récupération du gestionnaire de composants
    		xMultiComponentFactory = OOoUtils.getInterface(XMultiComponentFactory.class, objectInitial);
    		// Récupération des propriétés du gestionnaire de composants
    		XPropertySet xPropertySetMultiComponentFactory = OOoUtils.getInterface(XPropertySet.class, xMultiComponentFactory);
    		// Récupération du contexte par défaut de OOo distant
    		Object objectDefaultContext = xPropertySetMultiComponentFactory.getPropertyValue("DefaultContext");
    		// Association à un contexte de composant
    		xComponentContext = OOoUtils.getInterface(XComponentContext.class, objectDefaultContext);
    		// Association au service Desktop (service racine) avec le contexte configuré jusqu'ici
    		Object xDesktop = xMultiComponentFactory.createInstanceWithContext("com.sun.star.frame.Desktop", xComponentContext);
    		// Récupération d'une instance de l'implémentation de XComponentLoader qui permet le chargement d'un document
    		xComponentLoader = OOoUtils.getInterface(XComponentLoader.class, xDesktop);
    	}
    	
    	return xComponentLoader;
    }
    
    /**
     * Méthode permettant le chargement d'un nouveau XComponent
     * 
     * @param xComponentLoader Le XComponentLoader à partir duquel se fait le chargement
     * @param documentType Type de document à charger pour le XComponent
     * @param hidden Booléen indiquant si l'instance d'OOo lancée doit être cachée ou non
     * @return Le XComponent demandé, <code>null</code> en cas d'erreur
     * @throws IOException Exception lancée au chargement du XComponent
     * @throws IllegalArgumentException Exception lancée au chargement du XComponent
     */
    public static XComponent loadNewComponent(XComponentLoader xComponentLoader, DocumentType documentType,
    		boolean hidden) throws IOException, IllegalArgumentException{
    	XComponent xComponent = null;
    	
    	if(xComponentLoader != null  &&  documentType != null){
    		// on configure les propriétés du component à charger
			PropertyValue[] prop = new PropertyValue[1];
			prop[0] = new PropertyValue();
			prop[0].Name = "Hidden";
			prop[0].Value = hidden;
			// Chargement d'un nouveau document
			xComponent = xComponentLoader.loadComponentFromURL(documentType.getValue(), "_blank", 0, prop);
    	}
    	
    	return xComponent;
    }
    
    /**
     * Méthode permettant le chargement d'un XComponent existant
     * 
     * @param xComponentLoader Le XComponentLoader à partir duquel se fait le chargement
     * @param documentURL URL du document à charger pour le XComponent
     * @param hidden Booléen indiquant si l'instance d'OOo lancée doit être cachée ou non
     * @return Le XComponent demandé, <code>null</code> en cas d'erreur
     * @throws IOException Exception lancée au chargement du XComponent
     * @throws IllegalArgumentException Exception lancée au chargement du XComponent
     */
    public static XComponent loadComponent(XComponentLoader xComponentLoader, String documentURL,
    		boolean hidden) throws IOException, IllegalArgumentException{
    	XComponent xComponent = null;
    	
    	if(xComponentLoader != null  &&  documentURL != null){
    		// on configure les propriétés du component à charger
			PropertyValue[] prop = new PropertyValue[1];
			prop[0] = new PropertyValue();
			prop[0].Name = "Hidden";
			prop[0].Value = hidden;
			// Chargement d'un nouveau document
			xComponent = xComponentLoader.loadComponentFromURL(documentURL, "_blank", 0, prop);
    	}
    	
    	return xComponent;
    }
    
    /**
     * Méthode permettant de sauvegarder un document chargé via un XComponent
     * 
     * @param filePath Chemin du fichier à sauvegarder
     * @param fileName Nom du fichier à sauvegarder
     * @param xComponent XComponent source
     * @param saveFormat Format de sauvegarde
     * @param overwrite Booléen indiquant si on doit écraser l'éventuel fichier existant
     * @return <code>true</code> si la sauvegarde a réussi, <code>false</code> sinon
     */
    public static boolean saveDocument(String filePath, String fileName, XComponent xComponent, SaveFormat saveFormat, boolean overwrite){
    	boolean save = false;
    	
    	if(filePath != null  &&  fileName != null  &&  xComponent != null  &&  saveFormat != null){
			XStorable xStorable = OOoUtils.getInterface(XStorable.class, xComponent);
			XModel xModel = OOoUtils.getInterface(XModel.class, xComponent);
			XFrame xFrame = xModel.getCurrentController().getFrame();
			xFrame.activate(); 
			// Configuration des propriétés pour la sauvegarde
			PropertyValue[] propertyValue = new PropertyValue[2];
			propertyValue[0] = new PropertyValue();
			propertyValue[0].Name = "Overwrite";
			propertyValue[0].Value = overwrite;
			// Récupération du filtre associé au format de sauvegarde
			String filterName = saveFormat.getFilterName();
			if(filterName != null){
				propertyValue[1] = new PropertyValue();
				propertyValue[1].Name = "FilterName";
				propertyValue[1].Value = saveFormat.getFilterName();
			}
			// création de l'url de destination
			String url = new StringBuilder("file:///").append(filePath).
			append(fileName).append(".").append(saveFormat.
					getExtension()).toString();
			
			try{
				switch(saveFormat){
				case PDF :  xStorable.storeToURL(url, propertyValue);
							break;
				case ODS :
				case ODG :
				case ODT :
				case DOC :
				case SDA :
				case XLS :	xStorable.storeAsURL(url, propertyValue);
							break;
				}
				
			}catch(IOException ioe) {
				System.out.println("Exception à l'enregistrement : " + ioe.getMessage());
			}

			// Sauvegarde réussie
			save = true;
    	}
    	
    	return save;
    }
    
    /**
     * Méthode d'insertion du texte dans une cellule d'un tableau
     * 
     * @param xTable Table contenant la cellule à laquelle on souhaite accéder
     * @param cellName Nom OOo de la cellule
     * @param text Texte à insérer dans la cellule
     * @throws Exception
     */
	public static void insertIntoCell(XTextTable xTable, String cellName, String text, Color color) throws Exception{ 
		if(xTable != null  &&  cellName != null  &&  text != null  &&  color != null){
			// Récupération de l'interface XText de la cellule via son nom
			XText xCellText = OOoUtils.getInterface(XText.class, xTable.getCellByName(cellName));
			// Création d'un curseur texte
			XTextCursor xCellCursor = xCellText.createTextCursor();
			// Définition de la couleur d'écriture dans la cellule
			XPropertySet xCellCursorProps = OOoUtils.getInterface(XPropertySet.class, xCellCursor);
			xCellCursorProps.setPropertyValue("CharColor", Utils.ColorToIntegerValue(color));
			// Mise en place du texte dans la cellule
			xCellText.setString(text);
		}
	}
	
	/**
	 * Méthode d'insertion d'un texte dans un composant XText
	 * 
	 * @param xText Le composant XText au sein duquel se fait l'insertion
	 * @param txt Le texte à insérer
	 * @throws Exception
	 */
    public static void insertTextInXText(XText xText, String txt) throws Exception {
    	if(xText != null  &&  txt != null){
	        // Création d'un curseur texte pour mettre en forme le texte
	        XTextCursor xTextCursor = xText.createTextCursor();
	        XPropertySet xCursorProps = OOoUtils.getInterface(XPropertySet.class, xTextCursor);
	        xCursorProps.setPropertyValue("CharPosture", FontSlant.ITALIC);
	        xCursorProps.setPropertyValue("CharWeight", new Float(FontWeight.BOLD));
	        xText.setString(txt);
    	}
    }
    
    /**
     * Ajoute l'image pointée par imageUrl à xShapes
     * 
     * @param xMultiServiceFactory L'instance de XMultiServiceFactory
     * @param xShapes L'instance de XShapes 
     * @param imageUrl L'URL de l'image
     * @throws Exception
     */
    public static void addImagetoXShapes(XMultiServiceFactory xMultiServiceFactory, XShapes xShapes,
    		String imageUrl) throws Exception{
    	if(xShapes != null  &&  imageUrl != null){
	        URL resource = OOoUtils.class.getResource(imageUrl);
	        
	        if(resource != null  &&  resource.getPath() != null){
	        	// Construction de l'URL de l'image
	        	String graphicUrl = new StringBuilder("file://").append(resource.getPath().replace('\\', '/')).toString();
	        	// Récupération de l'interface GraphicObjectShape pour ajouter l'image 
	        	Object oGraphic = xMultiServiceFactory.createInstance("com.sun.star.drawing.GraphicObjectShape");
	        	// Configuration de l'objet Graphic
	        	XPropertySet xGraphicProps = OOoUtils.getInterface(XPropertySet.class, oGraphic); 
	        	xGraphicProps.setPropertyValue("GraphicURL", graphicUrl);
	        	xGraphicProps.setPropertyValue("AnchorType", TextContentAnchorType.AT_PARAGRAPH);
	        	// Sans adaptation du texte à l'image
	        	xGraphicProps.setPropertyValue("TextWrap", WrapTextMode.THROUGHT);
	        	// Image mise en arrière plan
	        	xGraphicProps.setPropertyValue("Opaque", false);
	        	// Récupération de l'interface XShape depuis l'objet graphique
	        	XShape xGraphicShape = OOoUtils.getInterface(XShape.class, oGraphic);
	        	// Configuration du positionnement et de la taille
	        	ImageIcon image = new ImageIcon(resource);
	        	xGraphicShape.setSize(new Size(image.getIconWidth() * 30, image.getIconHeight() * 30));
	        	xGraphicShape.setPosition(new Point(1500, 1000));
	        	// Ajout
	        	xShapes.add(xGraphicShape);
	        }
    	}
    }
    
    /**
     * Méthode permettant de fermer une instance OOo démarrée
     * via un XComponent passé en entrée
     * 
     * @param xComponent Le XComponent de l'instance OOo à fermer
     */
    public static void closeOrDisposeInstance(XComponent xComponent){
    	if(xComponent != null){
    		// on essaie de fermer le XComponent
			XCloseable xCloseable = OOoUtils.getInterface(XCloseable.class, xComponent);

			if(xCloseable != null){
				try {
					xCloseable.close(false);
				} catch (CloseVetoException cve) {
					System.out.println("Exception à la fermeture : " + cve.getMessage());
				}
			}else{
				// Libération du XComponent
				xComponent.dispose();
			}
    	}
    }
    
    /**
     * Méthode permettant l'insertion de value dans une feuille calc
     * xSpreadSheet. value peut être soit une valeur si isValue est à
     * true, soit une formule si isValue est à false
     * 
     * @param x Abscisse de la cellule
     * @param y Ordonnée de la cellule
     * @param value Valeur pour la cellule
     * @param xSpreadSheet Feuille de calcul calc
     * @param isValue Booléen indiquant s'il s'agit d'une valeur ou d'une formule
     * @throws IndexOutOfBoundsException Exception lancée si les coordonnées
     * de la cellule ne sont pas valides
     */
    public static void insertIntoCalcCell(int x, int y, Object value, XSpreadsheet xSpreadSheet, boolean isValue)
    			throws IndexOutOfBoundsException{    
    	if(x >= 0  &&  y >= 0  &&  value != null  &&  xSpreadSheet != null){
    		// Récupération de la cellule de coordonnées (x,y) dans la feuille de calcul xSpreadSheet
    		XCell xCell = xSpreadSheet.getCellByPosition(x, y);
    		
    		// Insertion en tant que valeur
    		if(isValue){
    			if(value instanceof Number){
    				xCell.setValue(((Number)value).doubleValue());
    			}else if(value instanceof String){
    				XText xCellText = OOoUtils.getInterface(XText.class, xCell);
    				xCellText.setString(value.toString());
    			}
    		}else{ // Insertion en tant que formule
    			if(value instanceof String){
    				xCell.setFormula(value.toString());
    			}
    		}
    		
    	}
    	
    }
    
    /**
     * Méthode permettant d'obtenir l'équivalent 
     * en char d'une colonne indexée avec un entier
     * 
     * @param col Valeur entière de la colonne
     * @return Valeur en char équivalente
     */
    public static char intColToCharCol(int col){
    	return (char)(col + 65);
    }

}
