/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.e4.examples.flickrviewer;

import org.eclipse.swt.SWT;
import org.eclipse.swt.animation.*;
import org.eclipse.swt.effects.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

public class Dock extends Composite {
	static final int MARGIN = 6;
	static final long SLIDE_DURATION = 600; 
	static final long ZOOM_DURATION = 200;
	Composite slider; 
	Composite thumbnails;
	Button next, previous;
	Point size, zoomSize;
	Color glowColor;
	ImageCanvas selected;
	
public Dock (Composite parent, int style) {
	super (parent, style);
	GridLayout layout = new GridLayout(3, false);
	layout.marginWidth = layout.marginHeight = 0;
	setLayout(layout);
	
	glowColor = new Color(getDisplay(), 102, 153, 204);
	
	previous = new Button(this, SWT.PUSH);
	previous.setText("<");
	previous.setToolTipText("Previous Page");
	previous.setEnabled(false);
	
	slider = new Composite(this, SWT.NONE);
	slider.setLayoutData(new GridData(GridData.FILL_BOTH));
	thumbnails = new Composite(slider, SWT.NONE);
	Listener listener = new Listener() {
		public void handleEvent(Event event) {
			Rectangle rect = thumbnails.getBounds();
			Rectangle sliderRect = slider.getBounds();
			previous.setEnabled(rect.x < 0);
			next.setEnabled(rect.width + rect.x > sliderRect.width);
		}
	};
	thumbnails.addListener(SWT.Move, listener);
	thumbnails.addListener(SWT.Resize, listener);
	
	next = new Button(this, SWT.PUSH);
	next.setText(">");
	next.setToolTipText("Next Page");
	next.setEnabled(false);
	
	next.addListener(SWT.Selection, new Listener() {
		public void handleEvent(Event event) {
			PropertyAnimation slide = new LinearPropertyAnimation();
			slide.setTarget(thumbnails);
			slide.setProperty("x");
			Rectangle sliderRect = slider.getBounds();
			Rectangle rect = thumbnails.getBounds();
			slide.setFrom(new Integer(rect.x));
			slide.setTo(new Integer(Math.max(sliderRect.width - rect.width, rect.x - sliderRect.width + size.x)));
			slide.setDuration(SLIDE_DURATION);
			slide.start(thumbnails);
		}
	});
	previous.addListener(SWT.Selection, new Listener() {
		public void handleEvent(Event event) {
			PropertyAnimation slide = new LinearPropertyAnimation();
			slide.setTarget(thumbnails);
			slide.setProperty("x");
			Rectangle sliderRect = slider.getBounds();
			Rectangle rect = thumbnails.getBounds();
			slide.setFrom(new Integer(rect.x));
			slide.setTo(new Integer(Math.min(0, rect.x + sliderRect.width - size.x)));
			slide.setDuration(SLIDE_DURATION);
			slide.start(thumbnails);
		}
	});
}

public void removeAll() {
	selected = null;
	Control[] children = thumbnails.getChildren();
	for (int i = 0; i < children.length; i++) {
		Control control = children[i];
		control.dispose();
	}
	thumbnails.setBounds(0, 0, 0, slider.getBounds().height);
}

public void add (Image image, String description, Object data) {
	final ImageCanvas label = new ImageCanvas(thumbnails, SWT.NONE);
	label.setImage(image);
	label.setToolTipText(description);
	label.setData(data);
	label.addListener(SWT.MouseEnter, new Listener() {
		public void handleEvent(Event event) {
			Display display = getDisplay(); 
			ImageCanvas control = (ImageCanvas)event.widget;
			control.setCursor(display.getSystemCursor(SWT.CURSOR_HAND));
			control.zoom(zoomSize, ZOOM_DURATION);
		}
	});
	label.addListener(SWT.MouseExit, new Listener() {
		public void handleEvent(Event event) {
			ImageCanvas control = (ImageCanvas)event.widget;
			control.setCursor(null);
			control.zoom(size, ZOOM_DURATION);
		}
	});
	label.addListener(SWT.Resize, new Listener() {
		public void handleEvent(Event event) {
			Control[] children = thumbnails.getChildren();
			int width = MARGIN;
			for (int i = 0; i < children.length; i++) {
				width += children[i].getSize().x + MARGIN;
			}
			int height = slider.getBounds().height;
			thumbnails.setSize(width, height);
			int x = MARGIN;
			for (int i = 0; i < children.length; i++) {
				Control child = children[i];
				Point size = child.getSize();
				child.setLocation(x, (height - size.y) / 2);
				x += size.x + MARGIN;
			}
		}
	});
	
	label.addListener(SWT.MouseDown, new Listener() {
		public void handleEvent(Event event) {
			setSelected((ImageCanvas)event.widget);
		}
	});
	label.setSize(size);
//	if (selected == null) setSelected(label);
}

void setSelected (ImageCanvas widget) {
	if (selected == widget) return;
	if (selected != null) {
		selected.setEffect(null);
	}
	selected = widget;
	GlowEffect effect = new GlowEffect();
	effect.setColor(glowColor);
	effect.setSize(10);
	selected.setEffect(effect);
	Event ev = new Event();
	ev.data = widget.getData();
	notifyListeners(SWT.Selection, ev);
}

public Point computeSize(int wHint, int hHint, boolean changed) {
	int width = 0;
	width += previous.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
	width += next.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
	width += (thumbnails.getChildren().length * (zoomSize.x + MARGIN)) + MARGIN;
	int height = zoomSize.y + MARGIN * 2;
	if (wHint != SWT.DEFAULT) width = wHint;
	if (hHint != SWT.DEFAULT) height = hHint;
	Rectangle trim = computeTrim(0, 0, width, height);
	return new Point(trim.width, trim.height);
}

public void setThumbSize(Point pt) {
	size = pt;
}

public void setZoomSize(Point pt) {
	zoomSize = pt;
}

}
