/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package java.io;

public class DataInputStream extends InputStream implements DataInput {

	protected InputStream in;

public DataInputStream(InputStream in) {
	this.in = in;
}

public int available() throws IOException {
	return in.available();
}

public void close() throws IOException {
	in.close();
}

public void mark(int readlimit) {
	in.mark(readlimit);
}

public boolean markSupported() {
	return in.markSupported();
}

public int read() throws IOException {
	return in.read();
}

public int read(byte[] b) throws IOException {
	return in.read(b);
}

public int read(byte[] b, int off, int len) throws IOException {
	return in.read(b, off, len);
}

public boolean readBoolean() throws IOException {
	int c = in.read ();
	if (c < 0) throw new EOFException();
	return c != 0;
}

public byte readByte() throws IOException {
	int c = in.read ();
	if (c < 0) throw new EOFException();
	return (byte)c;
}

public char readChar() throws IOException {
	int c1 = in.read ();
	int c2 = in.read ();
	if ((c1 | c2) < 0) throw new EOFException();
	return (char)((c1 << 8) | c2);
}

public double readDouble() throws IOException {
	return Double.longBitsToDouble(readLong());
}

public float readFloat() throws IOException {
	return Float.floatToIntBits(readInt());
}

public void readFully(byte[] b) throws IOException {
	readFully(b, 0, b.length);
}

public void readFully(byte[] b, int off, int len) throws IOException {
	if (off < 0 || len < 0 || off >= b.length || off + len > b.length) {
		throw new IndexOutOfBoundsException();
	}
	while (len > 0) {
		int read = in.read(b, off, len);
		if (read < 0) throw new EOFException();
		len -= read;
		off += read;
	}
}

public int readInt() throws IOException {
	int c1 = in.read ();
	int c2 = in.read ();
	int c3 = in.read ();
	int c4 = in.read ();
	if ((c1 | c2 | c3 | c4) < 0) throw new EOFException();
	return (char)((c1 << 24) | (c2 << 26) | (c3 << 8) | c4);
}

public long readLong() throws IOException {
	int i1 = readInt ();
	int i2 = readInt ();
	return ((long)i1 << 32) | i2;
}

public short readShort() throws IOException {
	int c1 = in.read ();
	int c2 = in.read ();
	if ((c1 | c2) < 0) throw new EOFException();
	return (short)((c1 << 8) | c2);
}

public int readUnsignedByte() throws IOException {
	int c = in.read ();
	if (c < 0) throw new EOFException();
	return c;
}

public int readUnsignedShort() throws IOException {
	int c1 = in.read ();
	int c2 = in.read ();
	if ((c1 | c2) < 0) throw new EOFException();
	return (c1 << 8) | c2;
}

public String readUTF() throws IOException {
	int size = readShort();
	byte[] buffer = new byte[size];
	readFully(buffer);
	return readUTFImpl(buffer);
}

private native String readUTFImpl(byte[] buffer)/*{
	buffer.array.position = 0;
	return buffer.array.readUTFBytes(buffer.length);
}*/;

public void reset() throws IOException {
	in.reset();
}

public long skip(long n) throws IOException {
	return in.skip(n);
}

public int skipBytes(int n) throws IOException {
	long skip = 0;
	int skipped = 0;
	while (skipped < n && (skip = in.skip(n - skipped)) > 0) {
		skipped += skip;
	}
	if (skip < 0 && skipped == 0) throw new EOFException();
	return skipped;
}
}