/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package java.lang;

public final class Character implements Comparable {

	private final char value;

	public static final int MIN_RADIX = 2;

	public static final int MAX_RADIX = 36;

	public static final char MIN_VALUE = '\u0000';

	public static final char MAX_VALUE = '\uFFFF';

	public static final Class TYPE = new Class("char");

	public static int digit(char c, int radix) {
		if (MIN_RADIX <= radix && radix <= MAX_RADIX) {
			if (c >= '0' && c < '0' + Math.min(radix, 10)) {
				return c - '0';
			}	
			if (c >= 'a' && c < (radix + 'a' - 10)) {
				return c - 'a' + 10;
			}	
			if (c >= 'A' && c < (radix + 'A' - 10)) {
				return c - 'A' + 10;
			}
		}
		return -1;
	}

	public static char forDigit(int digit, int radix) {
		if (radix < MIN_RADIX || radix > MAX_RADIX) {
			return 0;
		}
		if (digit < 0 || digit >= radix) {
			return 0;
		}
		if (digit < 10) {
			return (char) ('0' + digit);
		} else {
			return (char) ('a' + digit - 10);
		}
	}
	
	public static final byte DIRECTIONALITY_LEFT_TO_RIGHT = 0;
	public static final byte DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC = 2;
	public static byte getDirectionality(char ch) {
		return DIRECTIONALITY_LEFT_TO_RIGHT;
	}

	public static native boolean isDigit(char c) /*{
		return String.fromCharCode(c).match(/\d/) != null;
	}*/;

	public static native boolean isLetter(char c) /*{
		return String.fromCharCode(c).match(/[A-Z]/i) != null;
	}*/;

	public static native boolean isLetterOrDigit(char c) /*{
		return String.fromCharCode(c).match(/[A-Z\d]/i) != null;
	}*/;

	public static boolean isLowerCase(char c) {
		return toLowerCase(c) == c && isLetter(c);
	}

	public static boolean isSpace(char c) {
		switch (c) {
		case ' ':
			return true;
		case '\n':
			return true;
		case '\t':
			return true;
		case '\f':
			return true;
		case '\r':
			return true;
		default:
			return false;
		}
	}
	
	public static boolean isWhitespace(char ch) {
		return isSpace(ch);
	}
	
	public static boolean isJavaLetter(char ch) {
		return isLetter(ch) || ch == '$' || ch == '_';
	}
	
	public static boolean isJavaLetterOrDigit(char ch) {
		return isJavaLetter(ch) || isDigit(ch);  
	}
	
	public static boolean isJavaIdentifierStart(char ch) {
		return isJavaLetter(ch);
	}

	public static boolean isJavaIdentifierPart(char ch) {
		return isJavaLetterOrDigit(ch);
	}
	
	public static boolean isUpperCase(char c) {
		return toUpperCase(c) == c && isLetter(c);
	}

	public static native char toLowerCase(char c) /*{
		return String.fromCharCode(c).toLowerCase().charCodeAt(0);
	}*/;

	public static String toString(char x) {
		return String.valueOf(x);
	}

	public static native char toUpperCase(char c) /*{
		return String.fromCharCode(c).toUpperCase().charCodeAt(0);
	}*/;

	public Character(char value) {
		this.value = value;
	}

	public char charValue() {
		return value;
	}

	public int compareTo(Character c) {
		if (value < c.value) return -1;
		if (value > c.value) return 1;
		return 0;
	}

	public int compareTo(Object o) {
		return compareTo((Character) o);
	}

	public boolean equals(Object obj) {
		return (obj instanceof Character) && (((Character) obj).value == value);
	}

	public int hashCode() {
		return value;
	}

	public String toString() {
		return String.valueOf(value);
	}
}
