/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package java.util;

import java.io.InputStream;

public abstract class ResourceBundle {

	Locale locale;
	protected ResourceBundle parent;
	
public ResourceBundle() {
}

public static final ResourceBundle getBundle(String baseName) throws MissingResourceException {
	return getBundle(baseName, Locale.getDefault());
}

public static final ResourceBundle getBundle(String baseName, Locale locale) throws MissingResourceException {
	ResourceBundle bundle = null, previousBundle = null;
	String localeString = locale.toString();
	while (localeString.length() > 0) {
		ResourceBundle temp = findBundle(baseName, localeString);
		if (temp != null) {
			if (bundle == null) bundle = temp;
			if (previousBundle != null) previousBundle.setParent(temp);
			previousBundle = temp;
		}
		int index = localeString.lastIndexOf("_");
		if (index == -1) break;
		localeString = localeString.substring(0, index);
	}
	if (!locale.equals(Locale.getDefault())) {
		localeString = Locale.getDefault().toString();
		while (localeString.length() > 0) {
			ResourceBundle temp = findBundle(baseName, localeString);
			if (temp != null) {
				if (bundle == null) bundle = temp;
				if (previousBundle != null && previousBundle.parent == null) {
					previousBundle.setParent(temp);
				}
				previousBundle = temp;
			}
			int index = localeString.lastIndexOf("_");
			if (index == -1) break;
			localeString = localeString.substring(0, index);
		}
	}
	ResourceBundle temp = findBundle(baseName, "");
	if (temp != null) {
		if (bundle == null) bundle = temp;
		if (previousBundle != null) previousBundle.setParent(temp);
		previousBundle = temp;
	}
	if (bundle == null) {
		throw new MissingResourceException("Resource bundle not found", baseName, null);
	}
	return bundle;
}

static ResourceBundle findBundle(String baseName, String localeString) {
	ResourceBundle bundle = null;
	String name = baseName;
	if (localeString.length() != 0) name += "_" + localeString;
	try {
		Class clazz = Class.forName(name);
		bundle = (ResourceBundle)clazz.newInstance();
	} catch (Exception e) {
		InputStream stream = ClassLoader.getSystemResourceAsStream(name.replace('.', '/') + ".properties");
		try {
			System.out.println(stream);
			bundle = new PropertyResourceBundle(stream);
		} catch (Exception e1) {
		}
	}
	return bundle;
}

//public static final ResourceBundle getBundle(String baseName, Locale locale, ClassLoader loader) throws MissingResourceException {
//	return null;
//}

public abstract Enumeration<String> getKeys();

public Locale getLocale() {
	return locale;
}

public final Object getObject(String key) {
	if (key == null) throw new NullPointerException();
	ResourceBundle bundle = this;
	while (bundle != null) {
		Object result = bundle.handleGetObject(key);
		if (result != null) return result;
		bundle = bundle.parent;
	}
	throw new MissingResourceException("Missing key", null, key);
}

public final String getString(String key) {
	return (String)getObject(key);
}

public final String[] getStringArray(String key) {
	return (String[])getStringArray(key);
}

protected abstract Object handleGetObject(String key);

protected void setParent(ResourceBundle parent) {
	this.parent = parent;
}

}
