/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.build;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.preferences.IPreferencesService;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.IVMInstallType;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.swt.tools.actionscript.ActionScriptCorePlugin;
import org.eclipse.swt.tools.actionscript.debug.ActionScriptDebugEventDispatcher;
import org.eclipse.swt.tools.actionscript.debug.ActionScriptDebugTarget;
import org.eclipse.swt.tools.actionscript.debug.ActionScriptDebuggerCallbacks;

import flash.tools.debugger.Bootstrap;
import flash.tools.debugger.PlayerDebugException;
import flash.tools.debugger.Session;
import flash.tools.debugger.SessionManager;

public class ActionScriptLaunchConfigurationDelegate implements
		ILaunchConfigurationDelegate {

	public static final String ACTIONSCRIPT_APP = ActionScriptCorePlugin.PLUGIN_ID + ".actionscriptApplication";
	public static final String ATTR_PROJECT_NAME = IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME;
	public static final String ATTR_MAIN_TYPE_NAME = IJavaLaunchConfigurationConstants.ATTR_MAIN_TYPE_NAME;
	//TODO: e4 we currently consider all output directories relative to the specified project
	public static final String ATTR_OUTPUT_DIR_NAME = ActionScriptCorePlugin.PLUGIN_ID + ".outputDirName";
	public static final String ATTR_STATIC_LINK_OPTION = ActionScriptCorePlugin.PLUGIN_ID + ".linkOption";
	public static final String ATTR_COMPILE_ID = ActionScriptCorePlugin.PLUGIN_ID + ".compileId";
	public static final String ATTR_THEME_OPTION = ActionScriptCorePlugin.PLUGIN_ID + ".themeOption";
	public static final String ATTR_THEME_PATH = ActionScriptCorePlugin.PLUGIN_ID + ".themePath";

	public void launch(ILaunchConfiguration config, String mode,
			ILaunch launch, IProgressMonitor monitor) throws CoreException {
		
		String projectName = config.getAttribute(ATTR_PROJECT_NAME, "");
		String mainClass = config.getAttribute(ATTR_MAIN_TYPE_NAME, "");
		String outputDir = config.getAttribute(ATTR_OUTPUT_DIR_NAME, "");
		boolean staticLinking = config.getAttribute(ATTR_STATIC_LINK_OPTION, false);
		
		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		IProject projectResource = (IProject) workspaceRoot.findMember(projectName);
		if (projectResource == null) throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Problem Launching : Project doesn't exist"));
		IJavaProject project = JavaCore.create((IProject) projectResource);
		
		IType mainType = project.findType(mainClass);
		if (mainType == null) throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Problem Launching : Main class doesn't exist"));
		ActionScriptCorePlugin.DEPLOY = false;
		ActionScriptCorePlugin.CROSS_DOMAIN = false;
		ActionScriptCorePlugin.THEME = config.getAttribute(ATTR_THEME_OPTION, false);
		if (ActionScriptCorePlugin.THEME) {
			ActionScriptCorePlugin.THEME_URL = config.getAttribute(ATTR_THEME_PATH, "");
		}
		generateSWFFiles(project, mainClass, outputDir);
		buildSWF(project, mainClass, outputDir, config, staticLinking);
		
		File workingDir = new File(project.getProject().getLocation().toOSString() + File.separator + outputDir);
		String htmlFile = workingDir.getAbsolutePath() + File.separator + getClassName(mainClass) + ".html";
		IVMInstall vmInstall = JavaRuntime.computeVMInstall(config);
		IVMInstallType installType = vmInstall.getVMInstallType();
		if (installType instanceof ActionScriptVMInstallType) {
			ActionScriptVMInstallType asType = (ActionScriptVMInstallType) installType;
			String exec = asType.getExecutable(vmInstall.getInstallLocation());
			if (exec != null) {
				String url = ActionScriptCorePlugin.LAUNCH_URL + new Path(htmlFile).toPortableString();
				if (mode.equals(ILaunchManager.DEBUG_MODE)) {
					SessionManager sessionManager = Bootstrap.sessionManager();
					sessionManager.setDebuggerCallbacks(new ActionScriptDebuggerCallbacks(installType.getName(), exec));
					try {
						sessionManager.startListening();
						Session session = sessionManager.launch(url, null, true, null);
						session.bind();
						session.resume();
						ActionScriptDebugTarget target = new ActionScriptDebugTarget(session, launch, mainClass, exec);
						launch.addDebugTarget(target);
						launch.addProcess(target.process);
						
						new ActionScriptDebugEventDispatcher(target).schedule();
					} catch (IOException e) {
						e.printStackTrace();
					} catch (PlayerDebugException e) {
						e.printStackTrace();
					}
						 
				} else {
					Process process = DebugPlugin.exec(new String[]{exec, url}, workingDir);
					DebugPlugin.newProcess(launch, process, mainClass);
				}
				return;					
			} else {
				throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Unable to find SWF VM."));
			}
		}
		throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Invalid JRE type for ActionScript application."));
	}
	
	public void generateSWF (IJavaProject project, String mainClass, String outputDir, ILaunchConfiguration config, boolean staticLink) throws CoreException {
		buildSWF (project, mainClass, outputDir, config, staticLink);
	}
	
	void buildSWF (IJavaProject project, String mainClass, String outputDir, ILaunchConfiguration config, boolean staticLink) throws CoreException {
		try {
			final StringBuffer compc = new StringBuffer();
			compc.append("mxmlc -debug=");
			IPreferencesService service = Platform.getPreferencesService();
			boolean value = service.getBoolean(ActionScriptCorePlugin.PLUGIN_ID, ActionScriptCorePlugin.PREF_ENABLE_DEBUG_CHECK, false, null);
			compc.append(Boolean.valueOf(value));
			compc.append(" ");
			compc.append("-use-network=true");
			if (!staticLink) {
				compc.append(" -runtime-shared-library-path=");
				String flexSDK = System.getProperty(ActionScriptParticipant.FLEX_SDK);
				String swcPath = new Path(flexSDK + "/frameworks/libs/framework.swc").toOSString();
				compc.append(swcPath);
				String swfPath = findRSL(flexSDK, ".swf");
				String swzPath = findRSL(flexSDK, ".swz");
				if (swzPath != null) {
					compc.append(",");
					compc.append(ActionScriptCorePlugin.DEPLOY ? ActionScriptCorePlugin.DEPLOY_URL : ActionScriptCorePlugin.LAUNCH_URL);
					compc.append(swzPath.replace(File.separatorChar, '/'));
					compc.append(",");
				}
				compc.append(",");
				compc.append(ActionScriptCorePlugin.DEPLOY ? "" : ActionScriptCorePlugin.LAUNCH_URL);
				compc.append(swfPath.replace(File.separatorChar, '/'));
				if (ActionScriptCorePlugin.CROSS_DOMAIN) {
					compc.append(",");
					compc.append(ActionScriptCorePlugin.CROSS_DOMAIN_URL);
				}
			}
			IProject projectResource = project.getProject();
			IClasspathEntry[] entries = project.getRawClasspath();
			for (int i = 0; i < entries.length; i++) {
				IClasspathEntry entry = entries[i];
				if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER && entry.getPath().toPortableString().startsWith("org.eclipse.jdt.launching.JRE_CONTAINER")) {
					IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
					IProject jcl = (IProject)workspaceRoot.findMember("org.eclipse.swt.e4.jcl");
					if (jcl != null && jcl.isOpen()) {
						appendLibrary(compc, staticLink, jcl, projectResource, outputDir);
					} else {
						appendLibrary(compc, staticLink, ActionScriptCorePlugin.getDefault().getJCLLocation(), projectResource, outputDir);
					}
				}
			}
			for (int i = 0; i < entries.length; i++) {
				IClasspathEntry entry = entries[i];
				IPath path = entry.getPath();
				int kind = entry.getEntryKind();
				switch (kind) {
					case IClasspathEntry.CPE_SOURCE: {
						break;
					}
					case IClasspathEntry.CPE_PROJECT: {
						IWorkspaceRoot workspaceRoot = projectResource.getWorkspace().getRoot();
						IProject proj = (IProject) workspaceRoot.findMember(path);
						//TODO e4 - if the project is not open
						appendLibrary(compc, staticLink, proj, projectResource, outputDir);
						break;
					}
					case IClasspathEntry.CPE_VARIABLE: {
						IClasspathEntry resolvedEntry = JavaCore.getResolvedClasspathEntry(entry);
						appendLibrary(compc, staticLink, resolvedEntry.getPath(), projectResource, outputDir);
						break;
					}
					case IClasspathEntry.CPE_LIBRARY: {
						appendLibrary(compc, staticLink, path, projectResource, outputDir);
						break;
					}
					case IClasspathEntry.CPE_CONTAINER: {
						if (path.toPortableString().startsWith("org.eclipse.jdt.launching.JRE_CONTAINER")) break;
						IClasspathContainer container = JavaCore.getClasspathContainer(path, project);
						IClasspathEntry[] resolvedEntries = container.getClasspathEntries();
						for (int j = 0; j < resolvedEntries.length; j++) {
							IClasspathEntry resolvedEntry = resolvedEntries[j];
							if (resolvedEntry.getEntryKind() == IClasspathEntry.CPE_PROJECT) {
								IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
								IProject proj = (IProject)workspaceRoot.findMember(resolvedEntry.getPath());
								appendLibrary(compc, staticLink, proj, projectResource, outputDir);
							} else {
								appendLibrary(compc, staticLink, resolvedEntries[j].getPath(), projectResource, outputDir);
							}
						}
						break;
					}
				}
			}
			appendLibrary(compc, staticLink, projectResource, projectResource, outputDir);
			if (ActionScriptCorePlugin.THEME) {
				compc.append(" ");
				compc.append("-theme=");
				compc.append(ActionScriptCorePlugin.THEME_URL);
			}
			compc.append(" ");
			if(ActionScriptCorePlugin.DEPLOY){
				compc.append(outputDir + File.separator);
			} else {
				compc.append(projectResource.getLocation().toOSString() + File.separator + outputDir + File.separator);			
			}
			String simpleName = getClassName(mainClass);	
			compc.append(simpleName);
			compc.append("M.mxml");
			int count = 0;
			String result = null;
			do {
				String compcString = compc.toString();
				if (config != null) {
					Object existingID = config.getAttribute(ATTR_COMPILE_ID, (String) null);
					if (existingID != null) {
						result = ActionScriptCorePlugin.getDefault().getManager().sendCommand("info " + existingID);
						if (parseInfo (result).equals(compcString)) {
							compcString = "compile " + existingID;
						}		
					}  
				}
				result = ActionScriptCorePlugin.getDefault().getManager().sendCommand(compcString);
			} while (result == null && count++ < 3);
			if (result == null) throw new Exception("Out of Memory");
			String id = parseID (result);
			if (config != null) {
				if (id != null) { 
					ILaunchConfigurationWorkingCopy configWC = config.getWorkingCopy();
					configWC.setAttribute(ATTR_COMPILE_ID, id);
					config = configWC.doSave();
				}
			}
			if (!ActionScriptCorePlugin.DEPLOY) {
				projectResource.findMember(new Path(outputDir + File.separator + simpleName + "M.mxml")).delete(true, null);
				projectResource.findMember(new Path(outputDir)).refreshLocal(IResource.DEPTH_INFINITE, null);
			}
		} catch (Exception e) {
			throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Problem compiling SWC", e));
		}
	}

	void appendLibrary(StringBuffer compc, boolean staticLink, IPath path, IProject outputProject, String outputDir) throws CoreException, IOException {
		IPreferencesService service = Platform.getPreferencesService();
		String globalSWCPath = service.getString(ActionScriptCorePlugin.PLUGIN_ID, ActionScriptCorePlugin.PREF_SWC_PATH, null, null);
		if (globalSWCPath == null || globalSWCPath.length() == 0) return;
		String name = path.lastSegment();
		int index = name.lastIndexOf('.');
		if (index != -1) name = name.substring(0, index);
		File file = new File (globalSWCPath + File.separator + name + ".swc");
		if (file.exists()) {
			compc.append(staticLink ? " -library-path+=" : " -runtime-shared-library-path+=");
			String swcPath = file.getAbsolutePath();
			compc.append(swcPath);
			if (!staticLink) {
				String swfPath =  outputProject.getLocation().toOSString() + File.separator + outputDir + File.separator + name + ".swf";
				extractSWF (swcPath, outputDir, name + ".swf", outputProject);
				compc.append(",");
				compc.append(ActionScriptCorePlugin.DEPLOY ? ActionScriptCorePlugin.DEPLOY_URL : ActionScriptCorePlugin.LAUNCH_URL);
				compc.append(new Path(swfPath).toPortableString());
			}
		}
	}
	
	void appendLibrary (StringBuffer compc, boolean staticLink, IProject project, IProject outputProject, String outputDir) throws CoreException, IOException {
		compc.append(staticLink ? " -library-path+=" : " -runtime-shared-library-path+=");
		String swcPath = project.getLocation().toOSString() +  File.separator + ".buildas" + File.separator + project.getName() + ".swc";
		compc.append(swcPath);
		if (!staticLink) {
			String swfPath =  outputProject.getLocation().toOSString() + File.separator + outputDir + File.separator + project.getName() + ".swf";
			extractSWF (swcPath, outputDir, project.getName() + ".swf", outputProject);
			compc.append(",");
			if (ActionScriptCorePlugin.DEPLOY) {
				compc.append(project.getName() + ".swf");
			} else {
				compc.append(ActionScriptCorePlugin.LAUNCH_URL);
				compc.append(new Path(swfPath).toPortableString());
			}
			if (ActionScriptCorePlugin.CROSS_DOMAIN) {
				compc.append(",");
				compc.append(ActionScriptCorePlugin.CROSS_DOMAIN_URL);
			}
		}
	}
	private void extractSWF(String swcPath, String outputDir, String name, IProject project) throws CoreException, IOException {
		InputStream is = null;
		ZipFile zipFile = null;
		try {
			zipFile = new ZipFile(swcPath);
			ZipEntry zipEntry = zipFile.getEntry("library.swf");
			is = zipFile.getInputStream(zipEntry);
			if (ActionScriptCorePlugin.DEPLOY) {
				exportFile (outputDir, name, is);
			} else {
				createFile(project, outputDir, name, is);
			}
		} finally {
			if (zipFile != null) {
				try {
					zipFile.close();
				} catch (IOException e) {}
			}
		}
	}

	static void exportFile (String outputDir, String fileName, InputStream contentStream) {
		FileOutputStream os = null;
		String name = outputDir + File.separator + fileName;
		File file = new File(name);
		try {
			if (file.exists()) file.delete();
			if (contentStream != null) {
				int read;
				byte [] buffer = new byte [4096];
				os = new FileOutputStream (name);
				while ((read = contentStream.read (buffer)) != -1) {
					os.write(buffer, 0, read);
				}
				os.close ();
				contentStream.close ();
			}
			
		} catch (Throwable e) {
			ActionScriptCorePlugin.getDefault().getLog().log(
					new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID,
							"Failed to export file", e));
			try {
				if (os != null)
					os.close();
			} catch (IOException e1) {
			}
			try {
				if (contentStream != null)
					contentStream.close();
			} catch (IOException e1) {
			}
		}
	}
	private String findRSL (String flexSDK, String ext) {
		File[] files = new File(flexSDK + "/frameworks/rsls").listFiles();
		for (int i = 0; i < files.length; i++) {
			String name = files[i].getName();
			if (name.startsWith("framework_") && name.endsWith(ext)) {
				if (ActionScriptCorePlugin.DEPLOY) {
					return files[i].getAbsolutePath().substring(files[i].getAbsolutePath().lastIndexOf(File.separatorChar) + 1);
				} else {
					return files[i].getAbsolutePath();
				}
			}
		}
		return null;
	}

	public void generateSWFFiles (IJavaProject project, String mainClass, String outputDir) throws CoreException {
		IProject proj = project.getProject();
		buildMXMLFile(proj, mainClass, outputDir);
		buildHTMLFile(proj, mainClass, outputDir);
	}
	
	 String getClassName(String mainClass) {
			int index = mainClass.lastIndexOf('.');
			return index == -1 ? mainClass : mainClass.substring(index + 1);
		}

		void buildMXMLFile(IProject project, String mainClass, String outputDir) throws CoreException {
			StringBuffer buffer = new StringBuffer();
			String newLine = System.getProperty("line.separator");
			buffer.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
			buffer.append(newLine);
			buffer.append("<mx:Application xmlns:mx=\"http://www.adobe.com/2006/mxml\"  width=\"100%\" height=\"100%\" creationComplete=\"startUp();\">");	
			if (ActionScriptCorePlugin.THEME) {
				buffer.append(newLine);
				buffer.append("\t");	
				buffer.append("<mx:Style source=\"");
				buffer.append(ActionScriptCorePlugin.THEME_URL);
				buffer.append("\"/>");
			}
			buffer.append(newLine);
			buffer.append("\t");	
			buffer.append("<mx:Script><![CDATA[");
			buffer.append(newLine);
			buffer.append("\t\t");
			buffer.append("import ");
			buffer.append(mainClass);
			buffer.append(";");
			buffer.append(newLine);
			buffer.append("\t\t");
			buffer.append("import java.lang.JavaArray;");
			buffer.append(newLine);
			buffer.append("\t\t");
			buffer.append("import java.lang.System;");
			buffer.append(newLine);
			buffer.append(newLine);
			buffer.append("\t\t");
			buffer.append("private function startUp():void {");
			buffer.append(newLine);
			buffer.append("\t\t\t");
			buffer.append("java.lang.System.currentTimeMillis__();");
			buffer.append(newLine);
			buffer.append("\t\t\t");
			buffer.append("var result: JavaArray = new JavaArray(\"[Ljava/lang/String;\").lengths(0);");
			buffer.append(newLine);
			buffer.append("\t\t\t");
			buffer.append(mainClass);
			buffer.append(".main___3Ljava_lang_String_2(result);");
			buffer.append(newLine);
			buffer.append("\t\t");
			buffer.append("}");
			buffer.append(newLine);
			buffer.append("\t");
			buffer.append("]]></mx:Script>");               
			buffer.append(newLine);
			buffer.append("</mx:Application>");
			buffer.append(newLine);
		
			String simpleName = getClassName(mainClass);
			InputStream contentStream = new ByteArrayInputStream(buffer.toString().getBytes());
			if (ActionScriptCorePlugin.DEPLOY) {
				exportFile (outputDir, simpleName + "M.mxml", contentStream);
			} else {
				createFile(project, outputDir, simpleName + "M.mxml", contentStream);
			}
		}
		
		void buildHTMLFile(IProject project, String mainClass, String outputDir) throws CoreException {
			StringBuffer buffer = new StringBuffer();
			String newLine = System.getProperty("line.separator");
			buffer.append("<html lang=\"en\">");
			buffer.append(newLine);
			buffer.append("<head>");
			buffer.append(newLine);
			buffer.append("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\" />");
			buffer.append(newLine);
			buffer.append("<style>");
			buffer.append(newLine);
			buffer.append("body { margin: 0px; overflow:hidden }");
			buffer.append(newLine);
			buffer.append("</style>");
			buffer.append(newLine);
			buffer.append("<title>");
			buffer.append(mainClass);
			buffer.append("</title>");
			buffer.append(newLine);
			buffer.append("</head>");
			buffer.append(newLine);
			buffer.append("<body scroll='no'>");
			buffer.append(newLine);
			buffer.append("\t<object classid=\"clsid:D27CDB6E-AE6D-11cf-96B8-444553540000\" id=\"");
			buffer.append(mainClass);
			buffer.append("\" width=\"100%\" height=\"100%\" codebase=\"http://fpdownload.macromedia.com/get/flashplayer/current/swflash.cab\">");
			buffer.append(newLine);
			buffer.append("\t\t<param name=\"src\" value=\"");
			String path = "";
			if (!ActionScriptCorePlugin.DEPLOY) {				
				buffer.append(ActionScriptCorePlugin.LAUNCH_URL);
				path = project.getLocation().toPortableString() + "/" + outputDir + "/";
				buffer.append(path);
			}
			String simpleName = getClassName(mainClass);
			buffer.append(simpleName);
			buffer.append("M.swf\"/>");
			buffer.append(newLine);
			buffer.append("\t\t<param name=\"quality\" value=\"high\" />");
			buffer.append(newLine);
			buffer.append("\t\t<param name=\"bgcolor\" value=\"#869ca7\" />");
			buffer.append(newLine);
			buffer.append("\t\t<param name=\"allowScriptAccess\" value=\"sameDomain\" />");
			buffer.append(newLine);
			buffer.append("\t\t<embed src=\"");
			if (!ActionScriptCorePlugin.DEPLOY) {
				buffer.append(ActionScriptCorePlugin.LAUNCH_URL);
				buffer.append(path);
			}
			buffer.append(simpleName);
			buffer.append("M.swf\" name=\"");
			buffer.append(simpleName);
			buffer.append("\" quality=\"high\" bgcolor=\"#869ca7\" width=\"100%\" height=\"100%\" align=\"middle\" play=\"true\" loop=\"false\" quality=\"high\" allowScriptAccess=\"sameDomain\" type=\"application/x-shockwave-flash\" type=\"application/x-shockwave-flash\" pluginspage=\"http://www.adobe.com/go/getflashplayer\"/>");
			buffer.append(newLine);
			buffer.append("\t</object>");
			buffer.append(newLine);
			buffer.append("</body>");
			buffer.append(newLine);
			buffer.append("</html>");
			buffer.append(newLine);
			
			InputStream contentStream = new ByteArrayInputStream(buffer.toString().getBytes());
			if (ActionScriptCorePlugin.DEPLOY) {
				exportFile (outputDir, simpleName + ".html", contentStream);
			} else {
				createFile(project, outputDir, simpleName + ".html", contentStream);
			}
		}
			
		static void createFile (IProject project, String outputDir, String name, InputStream contentStream) throws CoreException {
			String path = outputDir + File.separator + name;
			IFile newFile = project.getFile(path);
			create(newFile.getParent());
			
			if (newFile.exists()) {
				newFile.setContents(contentStream, true, false, null);
			} else {
				newFile.create(contentStream, true, null);
			}
		}
	
		static void create(IContainer file) throws CoreException {
			if (file.exists()) return;
			switch (file.getType()) {
				case IResource.FOLDER:
					create(file.getParent());
					((IFolder)file).create(true, true, null);
			}
		}

		String parseID (String str) {
			String prefix = "fcsh: Assigned ";
			if (str.startsWith(prefix)){
				int startIndex = prefix.length();
				int endIndex = startIndex;
				while (endIndex < str.length()) {
					if (Character.isWhitespace(str.charAt(endIndex))) {
						return str.substring(startIndex, endIndex);
					}
					endIndex++;
				}
			}
			return null;
		}
		
		String parseInfo (String str) {
			String prefix = "mxmlc: ";
			int startIndex = str.indexOf(prefix) + prefix.length();
			if (startIndex != - 1) {
				int endIndex = str.length();
				int index = str.indexOf('\r', startIndex + prefix.length());
				if (index != -1) {
					endIndex = index - 1;
				} else {
					index = str.indexOf('\n', index + prefix.length());
					if (index != -1) {
						endIndex = index - 1;
					}
				}
				return "mxmlc " + str.substring(startIndex, endIndex);
			}
			return  "";
		}
}
