/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.build;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;

import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.preferences.IPreferencesService;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaModelMarker;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.compiler.BuildContext;
import org.eclipse.jdt.core.compiler.CompilationParticipant;
import org.eclipse.jdt.internal.compiler.as.Main;
import org.eclipse.swt.tools.actionscript.ActionScriptCorePlugin;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;


public class ActionScriptParticipant extends CompilationParticipant {
	
	public static final String COMPILE_ID_KEY = "swc_compile_id";
	public static final String FLEX_SDK = "flex.sdk";
	final String SOURCE_ID = "java_as";
	
	ArrayList sources;
	
	public void buildFinished(IJavaProject project) {
		if (sources == null) return;
		try {
			buildActionScript(project);
			buildSWC(project);
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}
	
	public void buildStarting(BuildContext[] files, boolean isBatch) {
		if (sources == null) sources = new ArrayList();
		for (int i = 0; i < files.length; i++) {
			BuildContext context = files[i];
			IFile file = context.getFile();
			sources.add(file.getLocation().toPortableString());
		}
	}
	
	public boolean isActive(IJavaProject project) {
		try {
			IProject proj = project.getProject();
			if (proj.getNature(ActionScriptCorePlugin.PLUGIN_ID + ".actionscriptnature") != null)
				return true;
			
			IProject[] projects = proj.getWorkspace().getRoot().getProjects();
			for (int i = 0; i < projects.length; i++) {
				if (projects[i].isOpen() && projects[i].getNature(ActionScriptCorePlugin.PLUGIN_ID + ".actionscriptnature") != null) {
					IJavaProject javaProject = JavaCore.create(projects[i]);
					if (javaProject != null) {
						if (javaProject.isOnClasspath(proj))
							return true;
					}
				}
			}
		} catch (CoreException e) {
		}
		return false;
	}
	
	void buildActionScript (IJavaProject project) throws CoreException {
		String root = project.getProject().getLocation().toPortableString() + "/.buildas";
		PrintWriter writer = null;
		try {
			StringBuffer sp = new StringBuffer(), cp = new StringBuffer(), bp = new StringBuffer();
			IClasspathEntry[] entries = project.getRawClasspath();
			for (int i = 0; i < entries.length; i++) {
				IClasspathEntry entry = entries[i];
				if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER && entry.getPath().toPortableString().startsWith("org.eclipse.jdt.launching.JRE_CONTAINER")) {
					IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
					IProject jcl = (IProject)workspaceRoot.findMember("org.eclipse.swt.e4.jcl");
					if (jcl != null && jcl.isOpen()) {
						IJavaProject javaProject = JavaCore.create(jcl.getProject());
						bp.append(workspaceRoot.findMember(javaProject.getOutputLocation()).getLocation().toPortableString());
					} else {
						bp.append(ActionScriptCorePlugin.getDefault().getJCLLocation().toPortableString());
					}
				}
			}
			for (int i = 0; i < entries.length; i++) {
				IClasspathEntry entry = entries[i];
				IPath path = entry.getPath();
				int kind = entry.getEntryKind();
				switch (kind) {
					case IClasspathEntry.CPE_SOURCE: {
						if (sp.length() > 0) sp.append(File.pathSeparatorChar);
						IResource resource = project.getProject().getWorkspace().getRoot().findMember(entry.getPath());
						sp.append(resource.getLocation().toPortableString());
						break;
					}
					case IClasspathEntry.CPE_PROJECT: {
						if (cp.length() > 0) cp.append(File.pathSeparator);
						//TODO e4 - if the project is closed?
						IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
						IResource resource = workspaceRoot.findMember(entry.getPath());
						IJavaProject javaProject = JavaCore.create((IProject) resource);
						cp.append(workspaceRoot.findMember(javaProject.getOutputLocation()).getLocation().toPortableString());
						break;
					}
					case IClasspathEntry.CPE_VARIABLE: {
						if (cp.length() > 0) cp.append(File.pathSeparatorChar);
						IClasspathEntry resolvedEntry = JavaCore.getResolvedClasspathEntry(entry);
						cp.append(resolvedEntry.getPath().toPortableString());
						break;
					}
					case IClasspathEntry.CPE_LIBRARY: {
						if (cp.length() > 0) cp.append(File.pathSeparatorChar);
						cp.append(entry.getPath().toPortableString());
						break;
					}
					case IClasspathEntry.CPE_CONTAINER: {
						if (path.toPortableString().startsWith("org.eclipse.jdt.launching.JRE_CONTAINER")) break;
						IClasspathContainer container = JavaCore.getClasspathContainer(path, project);
						IClasspathEntry[] resolvedEntries = container.getClasspathEntries();
						for (int j = 0; j < resolvedEntries.length; j++) {
							if (cp.length() > 0) cp.append(File.pathSeparator);
							IClasspathEntry resolvedEntry = resolvedEntries[j];
							if (resolvedEntry.getEntryKind() == IClasspathEntry.CPE_PROJECT) {
								IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
								IProject proj = (IProject)workspaceRoot.findMember(resolvedEntry.getPath());
								IJavaProject javaProject = JavaCore.create(proj);
								cp.append(workspaceRoot.findMember(javaProject.getOutputLocation()).getLocation().toPortableString());
							} else {
								cp.append(resolvedEntry.getPath().toPortableString());
							}
						}
						break;
					}
				}
			}
			String bin = root;
			if (cp.length() > 0) cp.append(File.pathSeparator);
			cp.append(bin);
			ArrayList args = new ArrayList();
			args.addAll(Arrays.asList(new String[]{
//				"-nowarn",
//				"-verbose",
				"-d", bin,
				"-bootclasspath", bp.toString(),
				"-cp", cp.toString(),
				"-log", root + "/log.xml",
				"-1.5",
				"-sourcepath", sp.toString(),
			}));
			args.addAll(sources);
			sources = null;
			IFolder newFolder = project.getProject().getFolder(".buildas");
			if (!newFolder.exists()) newFolder.create(true, true, null);
			writer = new PrintWriter(new BufferedOutputStream(new FileOutputStream(root + "/out.txt")));
			Main main = new Main(writer, writer, false, null);
			main.compile((String []) args.toArray(new String[args.size()]));
			writer.close();
			project.getProject().findMember(new Path(".buildas")).refreshLocal(IResource.DEPTH_INFINITE, null);
			createProblems(project, root);
		} catch (Exception e) {
			throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Problem compiling ActionScript", e));
		} finally {
			if (writer != null) writer.close();
		}
	}
	
	void appendLibrary(StringBuffer compc, IProject project) {
		compc.append(" -external-library-path+=");
		compc.append(project.getLocation().toPortableString() + "/.buildas/" + project.getName() + ".swc");
	}

	void appendLibrary(StringBuffer compc, IPath path) {
		IPreferencesService service = Platform.getPreferencesService();
		String globalSWCPath = service.getString(ActionScriptCorePlugin.PLUGIN_ID, ActionScriptCorePlugin.PREF_SWC_PATH, "", null);
		if (globalSWCPath == null || globalSWCPath.length() == 0) return;
		String name = path.lastSegment();
		int index = name.lastIndexOf('.');
		if (index != -1) name = name.substring(0, index);
		name += ".swc";
		File file = new File (globalSWCPath + File.separator + name);
		if (file.exists()) {
			compc.append(" -external-library-path+=");
			compc.append(new Path(file.getAbsolutePath()).toPortableString());
		}
	}
	
	public void buildSWC (IJavaProject project) throws CoreException {
		try {
			IProject projectResource = project.getProject();
			IResource resource = projectResource.findMember(new Path(".buildas"));
			if (resource != null && !resource.exists()) throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Missing .buildas folder.", null));
			resource.deleteMarkers(IMarker.PROBLEM, true, IResource.DEPTH_INFINITE);
			final String root = projectResource.getLocation().toOSString() + File.separator + ".buildas" + File.separator;
			generateResourceClass(project, ".buildas");
		
			final StringBuffer resourceSet = new StringBuffer();
			final StringBuffer compc = new StringBuffer();
			compc.append("compc ");
			compc.append("-warnings=false");
			compc.append(" ");
			compc.append("-debug=");
			IPreferencesService service = Platform.getPreferencesService();
			boolean value = service.getBoolean(ActionScriptCorePlugin.PLUGIN_ID, ActionScriptCorePlugin.PREF_ENABLE_DEBUG_CHECK, false, null);
			compc.append(Boolean.valueOf(value));

			compc.append(" -external-library-path+=");
			compc.append(new Path(System.getProperty(ActionScriptParticipant.FLEX_SDK)).toPortableString() + "/frameworks/libs/framework.swc");
			
			IClasspathEntry[] entries = project.getRawClasspath();
			for (int i = 0; i < entries.length; i++) {
				IClasspathEntry entry = entries[i];
				if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER && entry.getPath().toPortableString().startsWith("org.eclipse.jdt.launching.JRE_CONTAINER")) {
					IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
					IProject jcl = (IProject)workspaceRoot.findMember("org.eclipse.swt.e4.jcl");
					if (jcl != null && jcl.isOpen()) {
						appendLibrary(compc, jcl);
					} else {
						appendLibrary(compc, ActionScriptCorePlugin.getDefault().getJCLLocation());
					}
				}
			}
			for (int i = 0; i < entries.length; i++) {
				IClasspathEntry entry = entries[i];
				IPath path = entry.getPath();
				int kind = entry.getEntryKind();
				switch (kind) {
					case IClasspathEntry.CPE_SOURCE: {
						break;
					}
					case IClasspathEntry.CPE_PROJECT: {
						IWorkspaceRoot workspaceRoot = projectResource.getWorkspace().getRoot();
						IProject proj = (IProject) workspaceRoot.findMember(path);
						//TODO e4 - if the project is not open
						appendLibrary(compc, proj);
						break;
					}
					case IClasspathEntry.CPE_VARIABLE: {
						IClasspathEntry resolvedEntry = JavaCore.getResolvedClasspathEntry(entry);
						appendLibrary(compc, resolvedEntry.getPath());
						break;
					}
					case IClasspathEntry.CPE_LIBRARY: {
						appendLibrary(compc, path);
						break;
					}
					case IClasspathEntry.CPE_CONTAINER: {
						if (path.toPortableString().startsWith("org.eclipse.jdt.launching.JRE_CONTAINER")) break;
						IClasspathContainer container = JavaCore.getClasspathContainer(path, project);
						IClasspathEntry[] resolvedEntries = container.getClasspathEntries();
						for (int j = 0; j < resolvedEntries.length; j++) {
							IClasspathEntry resolvedEntry = resolvedEntries[j];
							if (resolvedEntry.getEntryKind() == IClasspathEntry.CPE_PROJECT) {
								IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
								IProject proj = (IProject)workspaceRoot.findMember(resolvedEntry.getPath());
								appendLibrary(compc, proj);
							} else {
								appendLibrary(compc, resolvedEntries[j].getPath());
							}
						}
						break;
					}
				}
			}
			
			compc.append(" ");
			compc.append("-source-path");
			compc.append(" ");
			compc.append(root);
			compc.append(" ");
			compc.append("-include-classes");
			resource.accept(new IResourceVisitor() {
				public boolean visit(IResource resource) throws CoreException {
					if ("as".equals(resource.getFileExtension())) {
						String tempString = resource.getLocation().toPortableString();
						tempString = tempString.substring(root.length());
						tempString = tempString.substring(0, tempString.length() - 3);
						tempString = tempString.replace('/','.');
						resourceSet.append(" ");
						resourceSet.append(tempString);
					}
					return true;
				}});
			compc.append(resourceSet);
			
			compc.append(" ");
			compc.append("-output=");
			compc.append(root + projectResource.getName() + ".swc");
			int count = 0;
			String result = null;
			do {
				String compcString = compc.toString();
				Object existingID = projectResource.getSessionProperty(new QualifiedName(ActionScriptCorePlugin.PLUGIN_ID,COMPILE_ID_KEY));
				if (existingID != null) {
					result = ActionScriptCorePlugin.getDefault().getManager().sendCommand("info " + existingID);
					if (parseInfo (result).equals(compcString)) {
						compcString = "compile " + existingID;
					}
				}
				result = ActionScriptCorePlugin.getDefault().getManager().sendCommand(compcString);
			} while (result == null && count++ < 3);
			if (result == null) throw new Exception("Out of Memory");
			String id = parseID (result);
			if (id != null) {
				projectResource.setSessionProperty(new QualifiedName(ActionScriptCorePlugin.PLUGIN_ID,COMPILE_ID_KEY), id);
			}
			projectResource.findMember(new Path(".buildas")).refreshLocal(IResource.DEPTH_INFINITE, null);
		} catch (Exception e) {
			throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Problem compiling SWC", e));
		}
	}
	
	void generateResourceClass (IJavaProject project, final String outputDir) throws JavaModelException, CoreException {
		final StringBuffer resourceFile = new StringBuffer();
		final String newLine = System.getProperty("line.separator");
		resourceFile.append("package {");
		resourceFile.append(newLine);
		resourceFile.append(newLine);
		//TODO: e4 Need to have a unique name for the class per project
		resourceFile.append("public class Resources {");
		resourceFile.append(newLine);
		resourceFile.append(newLine);
		
		final IProject projectResource = project.getProject();
		IResource outputLocation = projectResource.getWorkspace().getRoot().findMember(project.getOutputLocation());
		final String root = outputLocation.getLocation().toPortableString() + "/";
		 outputLocation.accept(new IResourceVisitor() {

			public boolean visit(IResource resource) {
				if (!"class".equals(resource.getFileExtension()) && resource instanceof IFile) {
					String tempString = resource.getLocation().toPortableString();
					tempString = tempString.substring(root.length());
					resourceFile.append("\t[Embed(source=\"");
					resourceFile.append(tempString);
					resourceFile.append("\", mimeType=\"application/octet-stream\")]");
					resourceFile.append(newLine);
					resourceFile.append("\tpublic var ");
					resourceFile.append(mangleResourceName(tempString));
					resourceFile.append(":Class;");
					resourceFile.append(newLine);
					resourceFile.append(newLine);
					try {
						 ActionScriptLaunchConfigurationDelegate.createFile(projectResource, outputDir + "/" + tempString.substring(0, tempString.length() - resource.getName().length()) , resource.getName(), ((IFile) resource).getContents());
					} catch (CoreException e) {
						e.printStackTrace();
					}
				}
				return true;
			}});
		 
		 resourceFile.append("}");
		 resourceFile.append(newLine);
		 resourceFile.append("}");
		 InputStream contentStream = new ByteArrayInputStream(resourceFile.toString().getBytes());
		 ActionScriptLaunchConfigurationDelegate.createFile(projectResource, outputDir, "Resources.as", contentStream);
	}
	
	void createProblems(IJavaProject project, String root) throws CoreException {
		IPreferencesService service = Platform.getPreferencesService();
		boolean showErrors = service.getBoolean(ActionScriptCorePlugin.PLUGIN_ID, ActionScriptCorePlugin.PREF_SHOW_ACTIONSCRIPT_ERRORS, false, null);
		if (showErrors) {
			try {
				InputStream is = new BufferedInputStream(new FileInputStream(root + "/log.xml"));
				Document doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new InputSource(is));
				is.close();
				IWorkspaceRoot workspaceRoot = project.getProject().getWorkspace().getRoot();
				String projPath = workspaceRoot.getLocation().toPortableString();
				NodeList sources = doc.getDocumentElement().getElementsByTagName("sources");
				for (int i = 0; i < sources.getLength(); i++) {
					NodeList src = ((Element)sources.item(i)).getElementsByTagName("source");
					for (int j = 0; j < src.getLength(); j++) {
						Element source = (Element)src.item(j);
						String path = source.getAttribute("path").replace('\\', '/');
						if (path.startsWith(projPath)) {
							path = path.substring(projPath.length());
						}
						IResource resource = workspaceRoot.findMember(new Path(path));
						boolean hasProblems = false;
						IMarker[] markers = resource.findMarkers(IJavaModelMarker.JAVA_MODEL_PROBLEM_MARKER, true, IResource.DEPTH_INFINITE);
						for (int m = 0; m < markers.length; m++) {
							IMarker marker = markers[m];
							if (SOURCE_ID.equals(marker.getAttribute(IMarker.SOURCE_ID))) {
								marker.delete();
							} else {
								Object severity = marker.getAttribute(IMarker.SEVERITY);
								hasProblems |= severity != null && ((Integer)severity).intValue() == IMarker.SEVERITY_ERROR;
							}
						}
						if (!hasProblems) {
							NodeList problems = source.getElementsByTagName("problems");
							for (int k = 0; k < problems.getLength(); k++) {
								NodeList problem = ((Element)problems.item(k)).getElementsByTagName("problem");
								for (int l = 0; l < problem.getLength(); l++) {
									Element node = (Element)problem.item(l);
									if (resource != null) {
										int start = Integer.parseInt(node.getAttribute("charStart"));
										int end = Integer.parseInt(node.getAttribute("charEnd")) + 1;
										String message = "[AS] " + ((Element)node.getElementsByTagName("message").item(0)).getAttribute("value");
										IMarker marker = resource.createMarker(IJavaModelMarker.JAVA_MODEL_PROBLEM_MARKER);
										int severity = "ERROR".equals(node.getAttribute("severity")) ? IMarker.SEVERITY_ERROR : IMarker.SEVERITY_WARNING;
										marker.setAttributes(
											new String[] {IMarker.MESSAGE, IMarker.SEVERITY, IMarker.CHAR_START, IMarker.CHAR_END, IMarker.SOURCE_ID},
											new Object[] {message, new Integer(severity), new Integer(start), new Integer(end), SOURCE_ID});
									}
								}
							}
						}
					}
				}
			} catch (Exception e) {
				throw new CoreException(new Status(IStatus.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Problem creating ActionScript problems", e));
			}
		}
	}
	
	public void cleanStarting(IJavaProject project) {
		if (!isActive(project)) return;
		sources = null;
		IResource resource = project.getProject().findMember(new Path(".buildas"));
		if (resource != null) {
			try {
				resource.delete(true, null);
			} catch (CoreException e) {
				e.printStackTrace();
			}
		}
	}
	
	String parseID (String str) {
		String prefix = "fcsh: Assigned ";
		if (str.startsWith(prefix)){
			int startIndex = prefix.length();
			int endIndex = startIndex;
			while (endIndex < str.length()) {
				if (Character.isWhitespace(str.charAt(endIndex))) {
					return str.substring(startIndex, endIndex);
				}
				endIndex++;
			}
		}
		return null;
	}
	
	String parseInfo (String str) {
		String prefix = "compc: ";
		int startIndex = str.indexOf(prefix) + prefix.length();
		if (startIndex != - 1) {
			int endIndex = str.length();
			int index = str.indexOf('\r', startIndex + prefix.length());
			if (index != -1) {
				endIndex = index - 1;
			} else {
				index = str.indexOf('\n', index + prefix.length());
				if (index != -1) {
					endIndex = index - 1;
				}
			}
			return "compc " + str.substring(startIndex, endIndex);
		}
		return  "";
	}
	
	String mangleResourceName (String tempString) {
		String mangledName = tempString.replace(' ','_');
		mangledName = mangledName.replace('/', '_');
		mangledName = mangledName.replace('.', '_');
		mangledName = mangledName.replace('-', '_');
		return mangledName;
	}
}
