/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.debug;

import java.util.HashMap;

import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.DebugElement;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IMemoryBlock;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStep;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.jdt.debug.core.IJavaLineBreakpoint;
import org.eclipse.swt.tools.actionscript.ActionScriptCorePlugin;
import org.eclipse.swt.tools.actionscript.debug.ActionScriptLineMapper.FileLocation;

import flash.tools.debugger.Location;
import flash.tools.debugger.NoResponseException;
import flash.tools.debugger.NotConnectedException;
import flash.tools.debugger.PlayerDebugException;
import flash.tools.debugger.Session;
import flash.tools.debugger.SourceFile;
import flash.tools.debugger.SwfInfo;

public class ActionScriptDebugTarget extends DebugElement implements IDebugTarget, IStep {
	
	Session session;
	ActionScriptThread thread;
	public ActionScriptProcess process;
	ILaunch launch;
	
	IBreakpointManager breakpointManager;
	boolean isStepping;
	
	String mainClass;
	String processName;
	
	HashMap<Integer,Location> locations;
	
	public ActionScriptDebugTarget(Session session, ILaunch launch, String mainClass, String processName) {
		super(null);
		this.session = session;
		this.launch = launch;
		this.mainClass = mainClass;
		this.processName = processName;
		thread = new ActionScriptThread(this);
		process = new ActionScriptProcess(this, launch, processName);
		breakpointManager = DebugPlugin.getDefault().getBreakpointManager();
		breakpointManager.addBreakpointListener(this);
	    locations = new HashMap<Integer, Location>();
		fireCreationEvent();
	}

	public String getName() throws DebugException {
		if (process.isTerminated()) return "<terminated>" + mainClass;
		return mainClass;
	}

	public IProcess getProcess() {
		return process;
	}

	public IThread[] getThreads() throws DebugException {
		if (isTerminated()) return new IThread[0];
		return new IThread[] {thread};
	}

	public boolean hasThreads() throws DebugException {
		return !isTerminated();
	}

	public boolean supportsBreakpoint(IBreakpoint breakpoint) {
		return breakpoint instanceof IJavaLineBreakpoint;
	}

	public IDebugTarget getDebugTarget() {
		return this;
	}

	public ILaunch getLaunch() {
		return launch;
	}

	public String getModelIdentifier() {
		return ActionScriptCorePlugin.PLUGIN_ID + ".debugModelPresentation";
	}

	public boolean canTerminate() {
		return !process.isTerminated();
	}

	public boolean isTerminated() {
		return process.isTerminated();
	}

	public void terminate() throws DebugException {
		fireTerminateEvent();
		process.terminate();
	}

	public boolean canResume() {
		if (!session.isConnected()) return false;
		try {
			return session.isSuspended();
		} catch (PlayerDebugException e) {
			//fixme
			e.printStackTrace();
		}
		return false;
	}

	public boolean canSuspend() {
		if (!session.isConnected()) return false;
		try {
			return !session.isSuspended();
		} catch (NotConnectedException e) {
			e.printStackTrace();
		}
		return false;
	}

	public boolean isSuspended() {
		try {
			if (isDisconnected()) return false;
			return session.isSuspended();
		} catch (NotConnectedException e) {
			e.printStackTrace();
		}
		return false;
	}

	public void resume() throws DebugException {
		try {
			session.resume();
			thread.fireResumeEvent(DebugEvent.CLIENT_REQUEST);
		} catch (PlayerDebugException e) {
			Status status = new Status(Status.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Error resuming target", e);
			throw new DebugException(status);
		}
	}

	public void suspend() throws DebugException {
		try {
			session.suspend();
			thread.fireSuspendEvent(DebugEvent.CLIENT_REQUEST);
		} catch (PlayerDebugException e) {
			Status status = new Status(Status.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Error suspending target", e);
			throw new DebugException(status);
		}
	}

	public void breakpointAdded(IBreakpoint breakpoint) {
		if (!(breakpoint instanceof IJavaLineBreakpoint)) return;
		if (isTerminated()) return;
		try {
			installBreakpoint((IJavaLineBreakpoint) breakpoint);
		} catch (PlayerDebugException e) {
			e.printStackTrace();
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	boolean installBreakpoint(IJavaLineBreakpoint breakpoint) throws PlayerDebugException, CoreException {
		ActionScriptLineMapper mapper = ActionScriptLineMapper.getInstance();
		FileLocation fileLocation = mapper.getActionScriptLocation(breakpoint);
		if (fileLocation == null) return false;
		SwfInfo[] swfs = session.getSwfs();
		for (int i = 0; i < swfs.length; i++) {
			SwfInfo swfInfo = swfs[i];
			SourceFile[] sourceFiles = swfInfo.getSourceList(session);
			for (int j = 0; j < sourceFiles.length; j++) {
				SourceFile sourceFile = sourceFiles[j];
				String fullPath = sourceFile.getFullPath();
				if (fullPath.equals(fileLocation.fileName)) {
					int id = sourceFile.getId();
					Location location = session.setBreakpoint(id, fileLocation.lineNumber);
					breakpoint.getMarker().setAttribute("ASID", id);
					locations.put(Integer.valueOf(id), location);
					return location != null;
				}
			}
		}
		return false;
		
	}

	public void breakpointChanged(IBreakpoint breakpoint, IMarkerDelta delta) {
		//FIXME
	}

	public void breakpointRemoved(IBreakpoint breakpoint, IMarkerDelta delta) {
		if (breakpoint instanceof IJavaLineBreakpoint) {
			try {
				int temp = breakpoint.getMarker().getAttribute("ASID", -1);
				if (temp != -1) {
					Location location = locations.get(Integer.valueOf(temp));
					if (location != null) {
						session.clearBreakpoint(location);
					}
				}
				breakpoint.getMarker().setAttribute("ASID", -1);
			} catch (CoreException e) {
				e.printStackTrace();
			} catch (NoResponseException e) {
				e.printStackTrace();
			} catch (NotConnectedException e) {
				e.printStackTrace();
			}
		}
	}

	public boolean canDisconnect() {
		return false;
	}

	public void disconnect() throws DebugException {
	}

	public boolean isDisconnected() {
		return !session.isConnected();
	}

	public IMemoryBlock getMemoryBlock(long startAddress, long length) throws DebugException {
		return null;
	}

	public boolean supportsStorageRetrieval() {
		return false;
	}

	public boolean canStepInto() {
		try {
			return session.isSuspended();
		} catch (NotConnectedException e) {
			return false;
		}
	}

	public boolean canStepOver() {
		try {
			return session.isSuspended();
		} catch (NotConnectedException e) {
			return false;
		}
	}

	public boolean canStepReturn() {
		try {
			return session.isSuspended();
		} catch (NotConnectedException e) {
			return false;
		}
	}

	public boolean isStepping() {
		return isStepping;
	}

	public void stepInto() throws DebugException {
		isStepping = true;
		try {
			session.stepInto();
			thread.fireResumeEvent(DebugEvent.STEP_INTO);
		} catch (PlayerDebugException e) {
			e.printStackTrace();
		}
	}

	public void stepOver() throws DebugException {
		isStepping = true;
		try {
			session.stepOver();
			thread.fireResumeEvent(DebugEvent.STEP_OVER);
		} catch (PlayerDebugException e) {
			e.printStackTrace();
		}
	}

	public void stepReturn() throws DebugException {
		isStepping = true;
		try {
			session.stepOut();
			thread.fireResumeEvent(DebugEvent.STEP_RETURN);
		} catch (PlayerDebugException e) {
			e.printStackTrace();
		}
	}

	void stepEnd() {
		isStepping = false;
		thread.fireSuspendEvent(DebugEvent.STEP_END);
	}
	
	public IBreakpointManager getBreakpointManager() {
		return breakpointManager;
	}

}

