/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.debug;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.DebugElement;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IRegisterGroup;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IVariable;
import org.eclipse.swt.tools.actionscript.ActionScriptCorePlugin;
import org.eclipse.swt.tools.actionscript.debug.ActionScriptLineMapper.FileLocation;

import flash.tools.debugger.Frame;
import flash.tools.debugger.PlayerDebugException;
import flash.tools.debugger.Variable;

public class ActionScriptFrame extends DebugElement implements IStackFrame {

	ActionScriptDebugTarget target;
	Frame frame;
	FileLocation fileLocation;
	
	ActionScriptFrame(ActionScriptDebugTarget target, Frame frame) {
		super(target);
		this.target = target;
		this.frame = frame;
	}
	
	public boolean equals(Object obj) {
		if (obj == this) return true;
		if (!(obj instanceof ActionScriptFrame)) return false;
		ActionScriptFrame temp = (ActionScriptFrame) obj;
		return frame == temp.frame;
	}
	 
	public int hashCode() {
		return frame.hashCode();
	}
	
	public int getCharEnd() throws DebugException {
		return -1;
	}

	public int getCharStart() throws DebugException {
		return -1;
	}

	public int getLineNumber() throws DebugException {
		initLocation (); 	
		return fileLocation != null ? fileLocation.lineNumber : 0;
	}

	public String getName() throws DebugException {
		String sig = frame.getCallSignature();
		sig = ActionScriptVariableValue.replace(sig, "::", ".");
		sig = sig.replace('/', '.');
		sig += "(";
		//TODO demangle signature and add parameter types
		sig += ")";
		return sig + " line: " + getLineNumber();
	}

	public String getSourceName() {
		initLocation (); 	
		return fileLocation != null ? fileLocation.qualifiedName : null;
	}
	
	private void initLocation() {
		if (fileLocation != null) return;
		try {
			fileLocation = ActionScriptLineMapper.getInstance().getJavaLocation(frame.getLocation());
		} catch (CoreException e) {
			e.printStackTrace();
		}
		
	}

	public IRegisterGroup[] getRegisterGroups() throws DebugException {
		return new IRegisterGroup[0];
	}

	public IThread getThread() {
		return target.thread;
	}

	public IVariable[] getVariables() throws DebugException {
		try {
			//FIXME getVariableList deprecated in favour of getLocals, but
			// the two functions don't return the same thing. Using deprecated
			// function for now, but this should be revisited.
			Variable[] locals = target.session.getVariableList();
//			Variable[] locals = frame.getLocals(target.session);
			int len = locals.length;
			IVariable[] result = new ActionScriptVariable[len];
			for (int i = 0; i < len; i++) {
				result[i] = new ActionScriptVariable(target, locals[i]);
			}
			return result;
		} catch (PlayerDebugException e) {
			Status status = new Status(Status.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Error getting variables", e);
			throw new DebugException(status);
		}
	}

	public boolean hasRegisterGroups() throws DebugException {
		return false;
	}

	public boolean hasVariables() throws DebugException {
		try {
			Variable[] locals = frame.getLocals(target.session);
			return locals.length > 0;
		} catch (PlayerDebugException e) {
			Status status = new Status(Status.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Error getting variables", e);
			throw new DebugException(status);
		} 
	}

	public IDebugTarget getDebugTarget() {
		return target;
	}

	public ILaunch getLaunch() {
		return target.getLaunch();
	}

	public String getModelIdentifier() {
		return target.getModelIdentifier();
	}

	public boolean canStepInto() {
		return target.canStepInto();
	}

	public boolean canStepOver() {
		return target.canStepOver();
	}

	public boolean canStepReturn() {
		return target.canStepReturn();
	}

	public boolean isStepping() {
		return target.isStepping();
	}

	public void stepInto() throws DebugException {
		target.stepInto();
	}

	public void stepOver() throws DebugException {
		target.stepOver();
	}

	public void stepReturn() throws DebugException {
		target.stepReturn();
	}

	public boolean canResume() {
		return target.canResume();
	}

	public boolean canSuspend() {
		return target.canSuspend();
	}

	public boolean isSuspended() {
		return target.isSuspended();
	}

	public void resume() throws DebugException {
		target.resume();
	}

	public void suspend() throws DebugException {
		target.suspend();
	}

	public boolean canTerminate() {
		return target.canTerminate();
	}

	public boolean isTerminated() {
		return target.isTerminated();
	}

	public void terminate() throws DebugException {
		target.terminate();
	}
}
