/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.tools.actionscript.debug;

import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IValue;
import org.eclipse.debug.core.model.IVariable;
import org.eclipse.swt.tools.actionscript.ActionScriptCorePlugin;

import flash.tools.debugger.PlayerDebugException;
import flash.tools.debugger.Value;
import flash.tools.debugger.Variable;
import flash.tools.debugger.VariableType;

public class ActionScriptVariableValue implements IValue {

	Value value;
	ActionScriptVariable variable;

	ActionScriptVariableValue(ActionScriptVariable variable, Value value) {
		this.variable = variable;
		this.value = value;
	}

	public String getReferenceTypeName() throws DebugException {
		return value.getClassName();
	}

	public String getValueString() throws DebugException {
		switch (value.getType()) {
			case VariableType.STRING:
				Object obj = value.getValueAsObject();
				return "\"" + obj + "\"";
			case VariableType.NUMBER:
				return value.getValueAsString();
			case VariableType.NULL:
				return "null";
			case VariableType.UNDEFINED:
				return value.getValueAsString();
			case VariableType.OBJECT:
				String name = value.getTypeName();
				name = replace(name, "::", ".");
				name = name.substring(0, name.lastIndexOf('@'));
				return name + " (id=" + value.getId() + ")";
				
		}
		return value.getValueAsString();
	}
	
	static String replace(String str, String match, String replacement) {
		StringBuffer buffer = new StringBuffer();
		int index, start = 0;
		while ((index = str.indexOf(match, start)) != -1) {
			buffer.append(str.substring(start, index));
			buffer.append(replacement);
			start = index + match.length();
		}
		buffer.append(str.substring(start));
		return buffer.toString();
	}

	public IVariable[] getVariables() throws DebugException {
		try {
			Variable[] members = value.getMembers(variable.target.session);
			ActionScriptVariable[] result = new ActionScriptVariable[members.length];
			for (int i = 0; i < members.length; i++) {
				result[i] = new ActionScriptVariable(variable.target, members[i]);
			}
			return result;
		} catch (PlayerDebugException e) {
			Status status = new Status(Status.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Error getting variables", e);
			throw new DebugException(status);
		} 
	}

	public boolean hasVariables() throws DebugException {
		try {
			return value.getMemberCount(variable.target.session) != 0;
		} catch (PlayerDebugException e) {
			Status status = new Status(Status.ERROR, ActionScriptCorePlugin.PLUGIN_ID, "Error getting variables", e);
			throw new DebugException(status);
		} 
	}

	public boolean isAllocated() throws DebugException {
		return true; //???
	}

	public IDebugTarget getDebugTarget() {
		return variable.getDebugTarget();
	}

	public ILaunch getLaunch() {
		return variable.getLaunch();
	}

	public String getModelIdentifier() {
		return variable.getModelIdentifier();
	}

	public Object getAdapter(Class adapter) {
		if (adapter == IValue.class) return this;
		return super.getClass();
	}

}
