﻿using System;
using System.Windows;
using System.Windows.Input;
using Microsoft.Surface.Presentation;
using Microsoft.Surface.Presentation.Manipulations;

namespace MonProjet
{
    /// <summary>
    /// Permet de scrutter des gestuelles simples sur des éléments graphique
    /// et de notifier par une commande de leurs déclenchements
    /// </summary>
    public static class GestureBehavior
    {
        #region Propriétés de dépendances

        /// <summary>
        /// Propriétée attachée pour définir la commande éxécutée si les condutions de gestuelles sont remplies
        /// </summary>        
        public static readonly DependencyProperty CommandProperty = DependencyProperty.RegisterAttached("Command", typeof(ICommand), typeof(GestureBehavior), new PropertyMetadata(null));

        /// <summary>
        /// Propriétée attachée pour définir le type de gestuelle attendue
        /// </summary>
        public static readonly DependencyProperty DeltaStepProperty = DependencyProperty.RegisterAttached("DeltaStep", typeof(double), typeof(GestureBehavior), new PropertyMetadata(double.NaN));

        /// <summary>
        /// Propriétée attachée pour activée la scrutation des gestuelles
        /// </summary>
        public static readonly DependencyProperty EnableProperty = DependencyProperty.RegisterAttached("Enable", typeof(bool), typeof(GestureBehavior), new PropertyMetadata(false, GestureBehavior.OnEnableChanged));

        /// <summary>
        /// Propriétée attachée pour définir le type de gestuelle attendue
        /// </summary>
        private static readonly DependencyProperty ProcessorProperty = DependencyProperty.RegisterAttached("Processor", typeof(Affine2DManipulationProcessor), typeof(GestureBehavior), new PropertyMetadata(null));

        #endregion

        #region Fonctions d'accessions des propriétés de dépendance

        /// <summary>
        /// Obtient la valeur de la propriété de dépenance Command
        /// </summary>
        [AttachedPropertyBrowsableForType(typeof(UIElement))]
        public static ICommand GetCommand(DependencyObject element)
        {
            return ((ICommand)element.GetValue(GestureBehavior.CommandProperty));
        }

        /// <summary>
        /// Définit la valeur de la propriété de dépenance Command
        /// </summary>
        public static void SetCommand(DependencyObject element, ICommand value)
        {
            element.SetValue(GestureBehavior.CommandProperty, value);
        }

        /// <summary>
        /// Obtient la valeur de la propriété de dépenance DeltaStep
        /// </summary>
        [AttachedPropertyBrowsableForType(typeof(UIElement))]
        public static double GetDeltaStep(DependencyObject element)
        {
            return ((double)element.GetValue(GestureBehavior.DeltaStepProperty));
        }

        /// <summary>
        /// Définit la valeur de la propriété de dépenance DeltaStep
        /// </summary>
        public static void SetDeltaStep(DependencyObject element, double value)
        {
            element.SetValue(GestureBehavior.DeltaStepProperty, value);
        }

        /// <summary>
        /// Obtient la valeur de la propriété de dépenance Enable
        /// </summary>
        [AttachedPropertyBrowsableForType(typeof(UIElement))]
        public static bool GetEnable(DependencyObject element)
        {
            return ((bool)element.GetValue(GestureBehavior.EnableProperty));
        }

        /// <summary>
        /// Définit la valeur de la propriété de dépenance Enable
        /// </summary>
        public static void SetEnable(DependencyObject element, bool value)
        {
            element.SetValue(GestureBehavior.EnableProperty, value);
        }

        /// <summary>
        /// Obtient la valeur de la propriété de dépenance Processor
        /// </summary>
        private static Affine2DManipulationProcessor GetProcessor(DependencyObject element)
        {
            return ((Affine2DManipulationProcessor)element.GetValue(GestureBehavior.ProcessorProperty));
        }

        /// <summary>
        /// Définit la valeur de la propriété de dépenance Processor
        /// </summary>
        private static void SetProcessor(DependencyObject element, Affine2DManipulationProcessor value)
        {
            element.SetValue(GestureBehavior.ProcessorProperty, value);
        }

        #endregion

        #region Fonctions privées

        /// <summary>
        /// Se produit sur un changement de la propriété de dépendance Enable
        /// </summary>
        /// <param name="element">Elémént graphique supportant cette modification</param>
        /// <param name="args">Argument de l'appel</param>
        private static void OnEnableChanged(DependencyObject element, DependencyPropertyChangedEventArgs args)
        {
            Affine2DManipulationProcessor processor = null;

            // Si on le remet à FALSE on libère les ressources

            if ((bool)args.OldValue && !((bool)args.NewValue))
            {
                processor = GestureBehavior.GetProcessor(element);

                if (element != null)
                    processor.Affine2DManipulationDelta -= new System.EventHandler<Affine2DOperationDeltaEventArgs>(GestureBehavior.Affine2DManipulationDelta);

                GestureBehavior.SetProcessor(element, null);
            }
            else if (element is UIElement)
            {
                // Création du processor pour l'écoute de la gestuelle

                processor = new Affine2DManipulationProcessor(Affine2DManipulations.Rotate | Affine2DManipulations.Scale | Affine2DManipulations.TranslateX | Affine2DManipulations.TranslateY, (UIElement)element);
                GestureBehavior.SetProcessor(element, processor);

                // On se branche sur les événements qui nous informe de la gestuelle

                processor.Affine2DManipulationDelta += new System.EventHandler<Affine2DOperationDeltaEventArgs>(GestureBehavior.Affine2DManipulationDelta);

                // On se branche sur les contacts pour scruter la gestuelle

                Contacts.AddContactDownHandler(element, GestureBehavior.ContactEventHandler);
            }
        }

        /// <summary>
        /// Se produit sur un changement de l'état des contacts en scrutations
        /// </summary>
        /// <param name="sender">Source de l'appel</param>
        /// <param name="e">Argument de l'appel</param>
        private static void Affine2DManipulationDelta(object sender, Affine2DOperationDeltaEventArgs e)
        {
            double deltaStep = GestureBehavior.GetDeltaStep(((Affine2DManipulationProcessor)sender).Container);
            ICommand command = GestureBehavior.GetCommand(((Affine2DManipulationProcessor)sender).Container);

            // Si pas de step ou de commande on ne fait rien

            if (double.IsNaN(deltaStep) && command != null)
                return;

            // Test selon le type de gestuelle scrutée

            Affine2DManipulationProcessor processor = GestureBehavior.GetProcessor(((Affine2DManipulationProcessor)sender).Container);

            if (Math.Abs(e.CumulativeTranslation.Length) > deltaStep)
            {
                command.Execute(null);
                processor.CompleteManipulation();
            }
        }

        /// <summary>
        /// Se produit sur la création d'un contact sur le contrôle
        /// </summary>
        /// <param name="sender">Source de l'appel</param>
        /// <param name="e">Argument de l'apel</param>
        private static void ContactEventHandler(object sender, ContactEventArgs e)
        {
            Affine2DManipulationProcessor processor = GestureBehavior.GetProcessor((DependencyObject)sender);

            if (processor != null)
                processor.BeginTrack(e.Contact);
        }

        #endregion
    }
}
