/*
 * Copyright (C) 2006 Proformatique - http://www.proformatique.com/
 * Written by Richard Braun <rbraun@proformatique.com>
 * Didactic IAX2 client.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <pthread.h>
#include <iaxclient.h>

static pthread_mutex_t reg_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t reg_cond = PTHREAD_COND_INITIALIZER;
static int registered = 0;
static int id;

static pthread_mutex_t calling_mutex = PTHREAD_MUTEX_INITIALIZER;
static int calling = 0;

static pthread_mutex_t hangup_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t hangup_cond = PTHREAD_COND_INITIALIZER;
static int hangup = 0;

static int
handle_levels(struct iaxc_ev_levels *levels)
{
  printf("input: %4f, output: %4f\n", levels->input, levels->output);
  return 0;
}

static int
handle_registration(struct iaxc_ev_registration *reg)
{
  pthread_mutex_lock(&reg_mutex);
  registered = reg->reply;
  id = reg->id;
  pthread_cond_broadcast(&reg_cond);
  pthread_mutex_unlock(&reg_mutex);
  return 0;
}

static int
handle_call_state(struct iaxc_ev_call_state *call)
{
  if (!(call->state & IAXC_CALL_STATE_ACTIVE))
    {
      pthread_mutex_lock(&hangup_mutex);
      hangup = 1;
      pthread_cond_broadcast(&hangup_cond);
      pthread_mutex_unlock(&hangup_mutex);
    }

  return 0;
}

static int
event_callback(iaxc_event event)
{
  switch (event.type)
    {
      case IAXC_EVENT_LEVELS :
        return handle_levels(&event.ev.levels);

      case IAXC_EVENT_REGISTRATION :
        return handle_registration(&event.ev.reg);

      case IAXC_EVENT_STATE :
        return handle_call_state(&event.ev.call);
    }

  return 0;
}

static void *
sigint_thread(void *sigset)
{
  int _calling, signum;

  /* Attente d'un ventuel SIGINT (Ctrl-C). */
  sigwait(sigset, &signum);

  /* Si un appel est en cours, on l'annule, sinon on termine le processus. */
  pthread_mutex_lock(&calling_mutex);
  _calling = calling;
  pthread_mutex_unlock(&calling_mutex);

  if (_calling)
    {
      pthread_mutex_lock(&hangup_mutex);
      hangup = 1;
      pthread_cond_broadcast(&hangup_cond);
      pthread_mutex_unlock(&hangup_mutex);
    }

  else
    exit(0);

  return NULL;
}

int
main(int argc, char *argv[])
{
  char *user, *pass, *host, *number, *dest;
  int error, dest_length;
  pthread_t thread;
  sigset_t sigset;

  if (argc != 5)
    {
      fprintf(stderr, "usage: iaxc <user> <pass> <host> <number>\n");
      return EXIT_FAILURE;
    }

  user = argv[1];
  pass = argv[2];
  host = argv[3];
  number = argv[4];

  /* Blocage du signal SIGINT pour qu'il soit gr par un thread ddi. */
  sigemptyset(&sigset);
  sigaddset(&sigset, SIGINT);
  sigprocmask(SIG_BLOCK, &sigset, NULL);
  error = pthread_create(&thread, NULL, sigint_thread, &sigset);

  if (error)
    {
      fprintf(stderr, "error: unable to start thread\n");
      return EXIT_FAILURE;
    }

  /* IAXClient gre le son et jusqu' un appel maximum. */
  error = iaxc_initialize(AUDIO_INTERNAL_PA, 1);

  if (error)
    return EXIT_FAILURE;

  iaxc_set_event_callback(event_callback);
  iaxc_start_processing_thread();
  iaxc_set_formats(IAXC_FORMAT_ALAW, (IAXC_FORMAT_ULAW
                                     | IAXC_FORMAT_ALAW));
  iaxc_register(user, pass, host);

  pthread_mutex_lock(&reg_mutex);

  while (!registered)
    pthread_cond_wait(&reg_cond, &reg_mutex);

  if (registered != IAXC_REGISTRATION_REPLY_ACK)
    {
      pthread_mutex_unlock(&reg_mutex);
      fprintf(stderr, "error: registration failed\n");
      return EXIT_FAILURE;
    }

  pthread_mutex_unlock(&reg_mutex);

  printf("registered on %s\n", host);

  dest_length = snprintf(NULL, 0, "%s:%s@%s/%s", user, pass, host, number);
  dest = malloc((dest_length + 1) * sizeof(char));

  if (dest == NULL)
    {
      fprintf(stderr, "error: memory allocation failed\n");
      return EXIT_FAILURE;
    }

  sprintf(dest, "%s:%s@%s/%s", user, pass, host, number);
  iaxc_call(dest);
  free(dest);
  printf("called %s\n", number);

  pthread_mutex_lock(&calling_mutex);
  calling = 1;
  pthread_mutex_unlock(&calling_mutex);

  pthread_mutex_lock(&hangup_mutex);

  while (!hangup)
    pthread_cond_wait(&hangup_cond, &hangup_mutex);

  pthread_mutex_unlock(&hangup_mutex);

  iaxc_shutdown();
  iaxc_unregister(id);
  return EXIT_SUCCESS;
}
