﻿using System;
using System.Linq;
using System.Collections.Generic;


namespace Programmez.StockEchange.Services
{
    /// <summary>
    /// Classe de gestion des indices financiers
    /// </summary>
    public class QuoteController
    {
        static object _lock = new object();
        static QuoteController _instance;

        /// <summary>
        /// Obtient l'instance singleton
        /// </summary>        
        public static QuoteController Instance
        {
            get
            {
                lock (_lock)
                {
                    if (_instance == null)
                    {
                        _instance = new QuoteController();
                    }
                }
                return QuoteController._instance;
            }
        }

        /// <summary>
        /// Créé un indice de manière asynchrone
        /// </summary>
        /// <param name="symbol">Le symbole.</param>
        /// <param name="callback">La méthode de rappel.</param>
        public void CreateNewQuoteAcync(string symbol, Action<Quote> callback){
            YahooFinanceService.Instance.GetQuotesDataAsync(symbol, (r) => CreateNewQuoteAcyncCompleted(symbol, r, callback));
        }

        /// <summary>
        /// S'exécute à l'issue de la création asynchrone d'un indice
        /// </summary>
        /// <param name="symbol">Le symbole.</param>
        /// <param name="r">Le résultat de la requête Yahoo.</param>
        /// <param name="callback">La méthode de rappel.</param>
        private void CreateNewQuoteAcyncCompleted(string symbol, YahooFinanceServiceResult r, Action<Quote> callback)
        {
            if (r.Quotes.Count > 0)
            {
                Quote quote = r.Quotes[0];

                // Persistance physique de l'indice 
                using (var quotesDataContext = new QuotesDataContext())
                {
                    // Verification d'un existence préalable
                    Quote previousQuote = quotesDataContext.Quotes.SingleOrDefault(pq => pq.Symbol == quote.Symbol);
                    if (previousQuote != null)
                    {
                        // L'indice existe => On l'attache au contexte
                        quotesDataContext.Quotes.Attach(quote, previousQuote);
                    }
                    else {
                        // L'indice n'existe pas => On le créé
                        quotesDataContext.Quotes.InsertOnSubmit(quote);
                    }                    
                    quotesDataContext.SubmitChanges();
                }

                // Invocation de la méthode de rappel
                callback(r.Quotes[0]);
            }
            else {
                // Invocation de la méthode de rappel
                callback(null);
            }
        }

        /// <summary>
        /// Obtient un indice stocké localement
        /// </summary>
        /// <param name="symbol">Le symbole.</param>
        /// <returns>L'indice</returns>
        public Quote GetQuote(string symbol)
        {
            Quote quote;
            using (var quotesDataContext = new QuotesDataContext())
            {
                quote = quotesDataContext.Quotes.SingleOrDefault(q => q.Symbol == symbol);
            }
            return quote;
        }

        /// <summary>
        /// Obtient tous les indices stockés localement
        /// </summary>
        /// <returns>Les indices</returns>
        public List<Quote> GetAllQuotes()
        {
            List<Quote> quotes;
            using (var quotesDataContext = new QuotesDataContext())
            {
                quotes = quotesDataContext.Quotes.ToList();
            }
            return quotes;
        }

        /// <summary>
        /// Rafraichit le cache local d'indices à partir du Web service Yahoo
        /// </summary>
        /// <param name="callback">La méthode de rappel.</param>
        public void RefreshQuotesAsync(Action callback)
        {
            List<string> allSymbols;

            // Récupération des indices en local
            using (var quotesDataContext = new QuotesDataContext())
            {
                allSymbols = quotesDataContext.Quotes.Select(q => q.Symbol).ToList();                
            }

            // Rafraichissement à partir du Web Service
            YahooFinanceService.Instance.GetQuotesDataAsync(allSymbols, (r) => RefreshQuotesAsyncCompleted(r, callback));
        }


        /// <summary>
        /// Rafraichit un indice à partir du Web Service Yahoo
        /// </summary>
        /// <param name="symbol">Le symbole.</param>
        /// <param name="callback">La méthode de rappel.</param>
        public void RefreshQuotesAsync(string symbol, Action callback)
        {
            YahooFinanceService.Instance.GetQuotesDataAsync(symbol, (r) => RefreshQuotesAsyncCompleted(r, callback));
        }

        /// <summary>
        /// Rafraichit une liste d'indices  à partir du Web Service Yahoo
        /// </summary>
        /// <param name="symbols">Les symboles.</param>
        /// <param name="callback">La méthode de rappel.</param>
        public void RefreshQuotesAsync(List<string> symbols, Action callback)
        {
            YahooFinanceService.Instance.GetQuotesDataAsync(symbols, (r) => RefreshQuotesAsyncCompleted(r, callback));
        }

        /// <summary>
        /// S'exécute lors de chaque invocation de la méthode d'actualisation du Web Service
        /// </summary>
        /// <param name="r">Le résultat du Web Service.</param>
        /// <param name="callback">La méthode de rappel.</param>
        private void RefreshQuotesAsyncCompleted(YahooFinanceServiceResult r, Action callback) {
            using (var quotesDataContext = new QuotesDataContext())
            {
                // Itération sur les indices récupérés
                foreach (Quote q in r.Quotes) {

                    // Récupération de l'indice local et mise à jour 
                    Quote previousQuote = quotesDataContext.Quotes.SingleOrDefault(pq => pq.Symbol == q.Symbol);
                    if (previousQuote != null) {
                        previousQuote.ChangeRealtime = q.ChangeRealtime;
                        previousQuote.Name = q.Name;
                        previousQuote.LastTradePriceOnly = q.LastTradePriceOnly;
                        previousQuote.Name = q.Name;
                        previousQuote.PercentChange = q.PercentChange;
                        previousQuote.PreviousClose = q.PreviousClose;
                        previousQuote.Volume = q.Volume;
                        previousQuote.YearHigh = q.YearHigh;
                        previousQuote.YearLow = q.YearLow;                        
                    }
                }

                // Enregistrement du résultat
                quotesDataContext.SubmitChanges();
            }

            // Invocation de la méthode de rappel
            if(callback !=null) callback();
        }
                
    }
}
