﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Net;
using System.Xml.Linq;

namespace Programmez.StockEchange.Services
{
    /// <summary>
    /// Classe d'accès au Web Service Yahoo
    /// </summary>
    public class YahooFinanceService
    {
        static object _lock = new object();
        static YahooFinanceService _instance;
        private const string BASE_WEB_QUOTE_URL = "http://query.yahooapis.com/v1/public/yql?q=" +
                                       "select * from yahoo.finance.quotes where symbol in ({0})" +
                                       "&env=store://datatables.org/alltableswithkeys";

        /// <summary>
        /// Obtient l'instance singleton
        /// </summary>      
        public static YahooFinanceService Instance
        {
            get
            {
                lock (_lock)
                {
                    if (_instance == null)
                    {
                        _instance = new YahooFinanceService();
                    }
                }
                return YahooFinanceService._instance;
            }
        }

        /// <summary>
        /// Constructeur
        /// </summary>
        public YahooFinanceService()
        {

        }

        /// <summary>
        /// Obtient un indice de manière asynchrone
        /// </summary>
        /// <param name="symbol">Le symbole.</param>
        /// <param name="callback">La méthode de rappel.</param>
        public void GetQuotesDataAsync(string symbol, Action<YahooFinanceServiceResult> callback)
        {
            this.GetQuotesDataAsync(new List<string>() { symbol }, callback);
        }

        /// <summary>
        /// Obtient un liste d'indices de manière asynchrone
        /// </summary>
        /// <param name="symbols">Les symboles.</param>
        /// <param name="callback">La méthode de rappel.</param>
        public void GetQuotesDataAsync(List<string> symbols, Action<YahooFinanceServiceResult> callback)
        {
            // Instanciation WebClient
            WebClient webClient = new WebClient();

            // Construction de la requête
            string symbolList = String.Join(",", symbols.Select(s => "\"" + s + "\"").ToArray());
            string url = string.Format(BASE_WEB_QUOTE_URL, symbolList);

            // Invocation de la requête HTTP
            webClient.DownloadStringCompleted += new DownloadStringCompletedEventHandler((s, e) => GetQuotesDataAsyncCompleted(e, callback));
            webClient.DownloadStringAsync(new Uri(url));
        }

        /// <summary>
        /// S'exécute à l'issue des requêtes Yahoo
        /// </summary>        
        /// <param name="e">L'instance résultat <see cref="System.Net.DownloadStringCompletedEventArgs"/> de la requête HTTP.</param>
        /// <param name="callback">La méthode de rappel.</param>
        private void GetQuotesDataAsyncCompleted(DownloadStringCompletedEventArgs e, Action<YahooFinanceServiceResult> callback)
        {
            if (e.Error == null) {

                // Parsing du résultat HTTP
                XDocument xmlDocument = XDocument.Parse(e.Result);
                XElement results = xmlDocument.Root.Element("results");

                // Instanciation de la classe de résultat YahooFinanceServiceResult
                YahooFinanceServiceResult result = new YahooFinanceServiceResult();

                // Construction de la collection d'indices
                foreach (XElement el in results.Elements("quote"))
                {
                    if (el.Element("ErrorIndicationreturnedforsymbolchangedinvalid") == null || String.IsNullOrEmpty(el.Element("ErrorIndicationreturnedforsymbolchangedinvalid").Value))
                    {
                        string symbol = el.Attribute("symbol").Value;

                        // Instanciation d'un indice et définition de ses propriétés
                        Quote quote = new Quote();
                        quote.Symbol = symbol;
                        quote.Name = el.Element("Name").Value;
                        quote.LastTradePriceOnly = GetDouble(el.Element("LastTradePriceOnly").Value);
                        quote.Volume = GetDouble(el.Element("Volume").Value);
                        quote.PreviousClose = GetDouble(el.Element("PreviousClose").Value);
                        quote.ChangeRealtime = GetDouble(el.Element("ChangeRealtime").Value);
                        quote.PercentChange = GetDouble(el.Element("PercentChange").Value);
                        quote.YearLow = GetDouble(el.Element("YearLow").Value);
                        quote.YearHigh = GetDouble(el.Element("YearHigh").Value);
                        result.Quotes.Add(quote);
                    }
                }

                // Invocation de la méthode de rappel
                if (callback != null)
                {
                    callback(result);
                }
            }
        }

        /// <summary>
        /// Convertit une chaine de valeur du fichier en double
        /// </summary>
        /// <param name="input">Chaine</param>
        /// <returns>La valeur convertie</returns>
        private static double? GetDouble(string input)
        {
            if (input == null) return null;

            input = input.Replace("%", "").Replace("+", "");

            double value;

            if (double.TryParse(input, NumberStyles.Any, new CultureInfo("en-US"), out value)) return value;
            return null;
        }

        /// <summary>
        /// Convertit une chaine de valeur du fichier en date
        /// </summary>
        /// <param name="input">Chaine</param>
        /// <returns>La valeur convertie</returns>
        private static DateTime? GetDateTime(string input)
        {
            if (input == null) return null;

            DateTime value;

            if (DateTime.TryParse(input, new CultureInfo("en-US"), DateTimeStyles.None, out value)) return value;
            return null;
        }
    }
}
