﻿using System.Windows;
using Microsoft.Phone.Scheduler;
using System;
using Programmez.StockEchange.Services;
using Microsoft.Phone.Shell;
using System.Linq;
using System.Globalization;

namespace Programmez.StockEchange.TaskAgent
{
    /// <summary>
    /// Agent planifié associé à l'application
    /// </summary>
    public class ScheduledAgent : ScheduledTaskAgent
    {
        private static volatile bool _classInitialized;

        /// <remarks>
        /// Constructeur
        /// </remarks>
        public ScheduledAgent()
        {
            if (!_classInitialized)
            {
                _classInitialized = true;
                // Inscription au gestionnaire d'erreur
                Deployment.Current.Dispatcher.BeginInvoke(delegate
                {
                    Application.Current.UnhandledException += ScheduledAgent_UnhandledException;
                });
            }
        }


        /// <summary>
        /// Gère l'évènement UnhandledException
        /// </summary>
        /// <param name="sender">La source de l'événement.</param>
        /// <param name="e">L'instance <see cref="System.Windows.ApplicationUnhandledExceptionEventArgs"/> contenant les données d'évènements.</param>
        private void ScheduledAgent_UnhandledException(object sender, ApplicationUnhandledExceptionEventArgs e)
        {
            if (System.Diagnostics.Debugger.IsAttached)
            {                
                System.Diagnostics.Debugger.Break();
            }
        }


        /// <summary>
        /// Appelée lors l'exécution planifiée d'une tâche
        /// </summary>
        /// <param name="task">L'objet <see cref="T:Microsoft.Phone.Scheduler.ScheduledTask"/> dont l'exécution est demandée.</param>
        protected override void OnInvoke(ScheduledTask task)
        {

            try
            {
                // Vérification de la tâche 
                if (task.Name == "TileUpdateTask")
                {
                    // Appel du rachaissement des indices (NotifyComplete() est appelée dans la méthode de retour)
                    QuoteController.Instance.RefreshQuotesAsync(() => { RefreshQuotesDataAsyncCompleted(task); });
                }
            }
            catch (Exception ex)
            {
                // Appel de la méthode de gestion d'erreur et de NotifyComplete()
                OnError(ex);
                NotifyComplete();
            }
        }

        /// <summary>
        /// Appelée lors d'une erreur
        /// </summary>
        /// <param name="ex">L'exception.</param>
        private void OnError(Exception ex)
        {
            // On ne fait rien
        }

        /// <summary>
        /// Exécutée lorsque que les indices ont été rafraichis
        /// </summary>
        /// <param name="task">La tâche planifiée.</param>
        private void RefreshQuotesDataAsyncCompleted(ScheduledTask task)
        {
            try
            {
                // Récupération de tous les indices stockées en local
                foreach (Quote quote in QuoteController.Instance.GetAllQuotes())
                {
                    // Vérification de l'existence d'une Tile associée à l'indice
                    ShellTile TileToFind = ShellTile.ActiveTiles.SingleOrDefault(t => t.NavigationUri.OriginalString.EndsWith("?s=" + quote.Symbol.ToString()));
                    if (TileToFind != null)
                    {
                        // Mise à jour des propriétés de la Tile
                        string value = "";
                        if (quote.LastTradePriceOnly.HasValue) value = Math.Round(quote.LastTradePriceOnly.Value, 3).ToString();
                        StandardTileData newTileData = new StandardTileData
                        {
                            Title = quote.Symbol,
                            BackContent = quote.Name.ToLower(),
                            BackTitle = value.ToString(new CultureInfo("fr-FR"))
                        };

                        // Mise à jour des fonds en fonction de la variation du cours de l'indice
                        if (quote.ChangeRealtime.HasValue && quote.ChangeRealtime.Value < 0)
                        {
                            newTileData.BackgroundImage = new Uri("/Assets/RedVariationIcon.png", UriKind.Relative);
                            newTileData.BackBackgroundImage = new Uri("/Assets/RedVariationIcon.png", UriKind.Relative);
                        }
                        else
                        {
                            newTileData.BackgroundImage = new Uri("/Assets/GreenVariationIcon.png", UriKind.Relative);
                            newTileData.BackBackgroundImage = new Uri("/Assets/GreenVariationIcon.png", UriKind.Relative);
                        }
                        TileToFind.Update(newTileData);
                    }
                }

                string taskName = task.Name;
#if DEBUG
                // Appel toutes les dix secondes en debug
                ScheduledActionService.LaunchForTest(taskName, TimeSpan.FromSeconds(10));
#endif

            }
            catch (Exception ex)
            {
                // Appel de la méthode de gestion d'erreur
                OnError(ex);
            }

            // Notification au système de la fin de l'exécution
            NotifyComplete();
        }
    }
}