﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using Microsoft.Phone.Controls;
using Programmez.StockEchange.Services;
using Microsoft.Phone.Shell;
using Microsoft.Phone.Scheduler;
using System.Globalization;

namespace Programmez.StockExchange.UI
{
    /// <summary>
    /// Classe de la page de détail
    /// </summary>
    public partial class QuoteDetails : PhoneApplicationPage
    {
        string _symbol;

        /// <summary>
        /// Obtient ou définit le symbole de l'indice en cours de consultation
        /// </summary>
        /// <value>Le symbole.</value>
        public string Symbol
        {
            get { return _symbol; }
            set { _symbol = value; }
        }

        /// <summary>
        /// Constructeur
        /// </summary>
        public QuoteDetails()
        {
            InitializeComponent();
            this.Loaded += new RoutedEventHandler(QuoteDetails_Loaded);
        }

        /// <summary>
        /// Gère l'événement Loaded de la page
        /// </summary>
        /// <param name="sender">La source de l'événement.</param>
        /// <param name="e">L'instance <see cref="System.Windows.RoutedEventArgs"/> contenant les données d'événements.</param>
        void QuoteDetails_Loaded(object sender, RoutedEventArgs e)
        {
           // Récupération du symbole et liaison de données
            this.Symbol = this.NavigationContext.QueryString["s"];
            this.BindQuoteAndRefreshTile();
        }

        /// <summary>
        /// Lie la page à l'indice stockée en local
        /// </summary>
        void BindQuoteAndRefreshTile() {
            
            // Affectation du DataContext
            this.DataContext = QuoteController.Instance.GetQuote(this.Symbol);
            
            // Verification de l'existence d'une Tile sur l'indice
            bool tileExists = ShellTile.ActiveTiles.Any(t => t.NavigationUri.OriginalString == this.NavigationService.Source.OriginalString);
            if (!tileExists)
            {
                // On active le bouton "épingle"
                ((ApplicationBarIconButton)this.ApplicationBar.Buttons[1]).IsEnabled = true;
            }
            else
            {
                // Si une Tile existe, on rafraichit systématiquement ses propriétés à partir des dernières valeurs disponibles
                ShellTile tileToUpdate = ShellTile.ActiveTiles.SingleOrDefault(t => t.NavigationUri.OriginalString == this.NavigationService.Source.OriginalString);
                Quote quote = this.DataContext as Quote;

                // Mise à jour des propriétés de la Tile
                string value = "";
                if (quote.LastTradePriceOnly.HasValue) value = Math.Round(quote.LastTradePriceOnly.Value, 3).ToString();
                StandardTileData newTileData = new StandardTileData
                {
                    Title = quote.Symbol,
                    BackContent = quote.Name.ToLower(),
                    BackTitle = value.ToString(new CultureInfo("fr-FR"))
                };

                // Mise à jour des fonds en fonction de la variation du cours de l'indice
                if (quote.ChangeRealtime.HasValue && quote.ChangeRealtime.Value < 0)
                {
                    newTileData.BackgroundImage = new Uri("/Assets/RedVariationIcon.png", UriKind.Relative);
                    newTileData.BackBackgroundImage = new Uri("/Assets/RedVariationIcon.png", UriKind.Relative);
                }
                else
                {
                    newTileData.BackgroundImage = new Uri("/Assets/GreenVariationIcon.png", UriKind.Relative);
                    newTileData.BackBackgroundImage = new Uri("/Assets/GreenVariationIcon.png", UriKind.Relative);
                }
                tileToUpdate.Update(newTileData);
            }

        }

        /// <summary>
        /// Méthode de rappel à l'issue d'une actualisation de valeurs
        /// </summary>
        private void RefreshQuoteAsyncCompleted()
        {
            this.BindQuoteAndRefreshTile();
        }

        /// <summary>
        /// Gère l'événement Click du bouton d'actualisation
        /// </summary>
        /// <param name="sender">La source de l'événement.</param>
        /// <param name="e">L'instance <see cref="System.EventArgs"/> contenant les données d'événements.</param>
        private void ApplicationBarIconButtonRefresh_Click(object sender, EventArgs e)
        {
            // Invocation de la méthode de rafraichissement de la base locale
            QuoteController.Instance.RefreshQuotesAsync(this.Symbol, RefreshQuoteAsyncCompleted);
        }

        /// <summary>
        /// Gère l'événement Click du bouton épingle
        /// </summary>
        /// <param name="sender">La source de l'événement.</param>
        /// <param name="e">L'instance <see cref="System.EventArgs"/> contenant les données d'événements.</param>
        private void ApplicationBarIconButtonPin_Click(object sender, EventArgs e)
        {
            Quote quote = this.DataContext as Quote;
            string value = "";
            if (quote.LastTradePriceOnly.HasValue) value = Math.Round(quote.LastTradePriceOnly.Value, 3).ToString();

            // Instanciation de l'objet de définition des propriétés de la Tile
            StandardTileData newTileData = new StandardTileData
            {
                Title = quote.Symbol,
                BackContent = quote.Name.ToLower(),
                BackTitle = value.ToString(new CultureInfo("fr-FR"))
            };

            // Mise à jour des fonds en fonction de la variation du cours de l'indice
            if (quote.ChangeRealtime.HasValue && quote.ChangeRealtime.Value < 0)
            {
                newTileData.BackgroundImage = new Uri("/Assets/RedVariationIcon.png", UriKind.Relative);
                newTileData.BackBackgroundImage = new Uri("/Assets/RedVariationIcon.png", UriKind.Relative);
            }
            else
            {
                newTileData.BackgroundImage = new Uri("/Assets/GreenVariationIcon.png", UriKind.Relative);
                newTileData.BackBackgroundImage = new Uri("/Assets/GreenVariationIcon.png", UriKind.Relative);
            }

            // Création de la Tile
            ShellTile.Create(this.NavigationService.Source, newTileData);

        }
    }
}