package com.programmez.android.speechbot.tts;

import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.speech.tts.TextToSpeech;

import com.programmez.android.speechbot.App;
import com.programmez.android.speechbot.Constants;

public class Tts {
  protected static Tts instance;

  public static Tts getInstance(final Context context, final boolean initialize) {
    if (instance == null || initialize) {
      instance = new Tts(context);
    }
    return instance;
  }

  protected boolean initialized;
  protected final Context context;
  private TextToSpeech ttsImpl;

  protected Tts(final Context context) {
    this.context = context;
    initTtsImpl();
  }

  public void checkTtsData(final Activity activity, final int requestCode) {
    final Intent checkIntent = new Intent();
    checkIntent.setAction(TextToSpeech.Engine.ACTION_CHECK_TTS_DATA);
    activity.startActivityForResult(checkIntent, requestCode);
  }

  public Locale[] obtainAvailableLocales() throws TtsNotInitializedException {
    checkTtsImpl();
    final Map<String, Set<Locale>> availableLocales = obtainAllTtsSupportedLocales();
    final Locale[] result = buildSortedLocales(availableLocales);
    return result;
  }

  public boolean onCheckTtsDataResult(final Activity activity,
      final int resultCode) {
    if (resultCode == TextToSpeech.Engine.CHECK_VOICE_DATA_PASS)
      return true;
    else {
      final Intent installIntent = new Intent();
      installIntent.setAction(TextToSpeech.Engine.ACTION_INSTALL_TTS_DATA);
      activity.startActivity(installIntent);
      return false;
    }
  }

  public void setLanguage(final String language)
      throws TtsNotInitializedException {
    checkTtsImpl();
    ttsImpl.setLanguage(language == null || "".equals(language) ? Locale
        .getDefault() : new Locale(language));
  }

  public void setSpeechRate(final float speechRate) {
    ttsImpl.setSpeechRate(speechRate);
  }

  public void shutdown() {
    App.log("tts shutdown");
    shutdownTtsImpl();
    initialized = false;
    instance = null;
  }

  public void shutdownTtsImpl() {
    ttsImpl.shutdown();
  }

  public void speak(final String text, final int queueMode)
      throws TtsNotInitializedException {
    checkTtsImpl();
    ttsImpl.speak(text,
        queueMode == Constants.TTS_QUEUE_ADD ? TextToSpeech.QUEUE_ADD
            : TextToSpeech.QUEUE_FLUSH, null);
  }

  protected Locale[] buildSortedLocales(
      final Map<String, Set<Locale>> availableLocales) {
    final TreeSet<Locale> result = new TreeSet<Locale>(
        new Comparator<Locale>() {
          public int compare(final Locale l1, final Locale l2) {
            return l1 == l2 ? 0 : (l1 == null ? 1 : (l2 == null ? -1 : l1
                .getDisplayName().compareTo(l2.getDisplayName())));
          }
        });
    final Iterator<String> i = availableLocales.keySet().iterator();
    while (i.hasNext()) {
      final String lang = i.next();
      final Set<Locale> set = availableLocales.get(lang);
      final int nb = set.size();
      final Iterator<Locale> j = set.iterator();
      while (j.hasNext()) {
        final Locale locale = j.next();
        if ((nb == 2 && !"".equals(locale.getCountry()))
            || (nb > 2 && "".equals(locale.getCountry()))) {
          continue;
        }
        result.add(locale);
      }
    }
    return result.toArray(new Locale[result.size()]);
  }

  protected void checkTtsImpl() throws TtsNotInitializedException {
    if (getTtsImpl() == null || !initialized)
      throw new TtsNotInitializedException();
  }

  protected Object getTtsImpl() {
    return ttsImpl;
  }

  protected void initTtsImpl() {
    ttsImpl = new TextToSpeech(context, new TextToSpeech.OnInitListener() {
      public void onInit(final int status) {
        App.log("tts initialized with status=" + status);
        initialized = true;
      }
    });
  }

  protected Map<String, Set<Locale>> obtainAllTtsSupportedLocales() {
    final Locale[] systemLocales = Locale.getAvailableLocales();
    final Map<String, Set<Locale>> result = new HashMap<String, Set<Locale>>();
    for (int i = 0; i < systemLocales.length; i++) {
      final Locale systemLocale = systemLocales[i];
      final int languageAvailable = ttsImpl.isLanguageAvailable(systemLocale);
      final Locale locale;
      switch (languageAvailable) {
      case TextToSpeech.LANG_MISSING_DATA:
      case TextToSpeech.LANG_NOT_SUPPORTED:
        continue;
      case TextToSpeech.LANG_COUNTRY_VAR_AVAILABLE:
        locale = systemLocale;
        break;
      case TextToSpeech.LANG_COUNTRY_AVAILABLE:
        locale = new Locale(systemLocale.getLanguage(), systemLocale
            .getCountry());
        break;
      case TextToSpeech.LANG_AVAILABLE:
        locale = new Locale(systemLocale.getLanguage());
        break;
      default:
        locale = null;
      }
      if (locale != null) {
        Set<Locale> set = result.get(locale.getLanguage());
        if (set == null) {
          set = new HashSet<Locale>();
          result.put(locale.getLanguage(), set);
        }
        set.add(locale);
      }
    }
    return result;
  }
}
