package com.programmez.android.speechbot.tts;

import java.util.Comparator;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import android.app.Activity;
import android.content.Context;

import com.programmez.android.speechbot.App;

public abstract class Tts {
  protected static Tts instance;

  public static Tts getInstance(final Context context, final boolean initialize) {
    if (instance == null || initialize) {
      try {
        instance = new TtsDonut(context);
      } catch (final VerifyError e) {
        instance = new TtsCupcake(context);
      }
    }
    App.log("tts instance : " + instance.getClass().getSimpleName());
    return instance;
  }

  protected boolean initialized;
  protected final Context context;

  protected Tts(final Context context) {
    this.context = context;
    initTtsImpl();
  }

  public abstract void checkTtsData(final Activity activity, int requestCode);

  public Locale[] obtainAvailableLocales() throws TtsNotInitializedException {
    checkTtsImpl();
    final Map<String, Set<Locale>> availableLocales = obtainAllTtsSupportedLocales();
    final Locale[] result = buildSortedLocales(availableLocales);
    return result;
  }

  public abstract boolean onCheckTtsDataResult(final Activity activity,
      final int resultCode);

  public void setLanguage(final String language)
      throws TtsNotInitializedException {
    checkTtsImpl();
    setLanguageImpl(language == null || "".equals(language) ? Locale
        .getDefault() : new Locale(language));
  }

  public abstract void setSpeechRate(final float speechRate);

  public void shutdown() {
    App.log("tts shutdown");
    shutdownTtsImpl();
    initialized = false;
    instance = null;
  }

  public void speak(final String text, final int queueMode)
      throws TtsNotInitializedException {
    checkTtsImpl();
    speakImpl(text, queueMode);
  }

  protected Locale[] buildSortedLocales(
      final Map<String, Set<Locale>> availableLocales) {
    final TreeSet<Locale> result = new TreeSet<Locale>(
        new Comparator<Locale>() {
          public int compare(final Locale l1, final Locale l2) {
            return l1 == l2 ? 0 : (l1 == null ? 1 : (l2 == null ? -1 : l1
                .getDisplayName().compareTo(l2.getDisplayName())));
          }
        });
    final Iterator<String> i = availableLocales.keySet().iterator();
    while (i.hasNext()) {
      final String lang = i.next();
      final Set<Locale> set = availableLocales.get(lang);
      final int nb = set.size();
      final Iterator<Locale> j = set.iterator();
      while (j.hasNext()) {
        final Locale locale = j.next();
        if ((nb == 2 && !"".equals(locale.getCountry()))
            || (nb > 2 && "".equals(locale.getCountry()))) {
          continue;
        }
        result.add(locale);
      }
    }
    return result.toArray(new Locale[result.size()]);
  }

  protected void checkTtsImpl() throws TtsNotInitializedException {
    if (getTtsImpl() == null || !initialized)
      throw new TtsNotInitializedException();
  }

  protected abstract Object getTtsImpl();

  protected abstract void initTtsImpl();

  protected abstract Map<String, Set<Locale>> obtainAllTtsSupportedLocales();

  protected abstract void setLanguageImpl(Locale locale);

  protected abstract void shutdownTtsImpl();

  protected abstract void speakImpl(String text, int queueMode);

}
