#include <QtGui>

#include "ptray.h"
#include "pconfigdialog.h"
#include "pxmlhandler.h"

PTray::PTray() : PNoteEditor()
{
    warnOnClose=false;
    readSettings();
    createActions();
    createTrayIcon();
    
    connect(trayIcon, SIGNAL(messageClicked()), this, SLOT(messageClicked()));
    connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
            this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));
    connect(this, SIGNAL(noteTitleChanged(const QString&)), 
	    this, SLOT(on_noteTitleChanged(const QString&)) );
    connect(this, SIGNAL(noteTextChanged(const QString&)), 
	    this, SLOT(on_noteTextChanged(const QString&)) );
    trayIcon->show();
//     QRect trect = trayIcon->geometry();
//     setGeometry(trect.x()-300, trect.top()-210,300,200);
    setVisible(true);
    setNoteTitle("");
    setNoteText("");
    loadRssFile(notesFileName);
}

void PTray::loadRssFile(const QString &fileName){
    if(QFile::exists(fileName)){
	QFile file(fileName);
	QXmlInputSource inputSource(&file);
	QXmlSimpleReader reader;
	QList<QAction *> list;
	PXmlHandler handler(&list);
	reader.setContentHandler(&handler);
	reader.setErrorHandler(&handler);
	reader.parse(inputSource);
	for(int i=0; i < list.count(); i++){
	    QAction *act = list.at(i);
	    addNoteToHistory(act->text(), act->toolTip());
	}
    }
}

void PTray::writeNotes(){
    QFile file(notesFileName);
    if (!file.open(QFile::WriteOnly | QFile::Text)) {
	QMessageBox::warning(this, tr("Quick Desktop Notes"),
			    tr("Cannot write file %1:\n%2.")
			    .arg(notesFileName)
			    .arg(file.errorString()));
    }
    else{
	QTextStream out(&file);
	out.setCodec("UTF-8");
	QApplication::setOverrideCursor(Qt::WaitCursor);
	QApplication::restoreOverrideCursor();
	out << "<?xml version=\"1.0\"?>\n<rss version=\"2.0\">\n<channel>\n";
	out << "<title>Quick Desktop Notes</title>\n";
	out << "<link>file://" << notesFileName << "</link>\n";
	out << "<description>Quick Desktop Notes data file.</description>\n";
	out << "<language>" << QLocale::system().name() << "</language>\n";
	out << "<pubDate>" << QDateTime::currentDateTime().toString(Qt::TextDate) << "</pubDate>\n";
	out << "<lastBuildDate>" << QDateTime::currentDateTime().toString(Qt::TextDate) << "</lastBuildDate>\n";
	out << "<docs></docs>\n";
	out << "<generator>QuickDesktopNotes 1.0</generator>\n";
	out << "<managingEditor></managingEditor>\n";
	out << "<webMaster></webMaster>\n";
	out << "<ttl></ttl>\n\n";
	
	QAction *ita;
	foreach (ita, history->actions()){
	    if(!ita->text().isEmpty() ){
		out << "<item>\n<title>" << Qt::escape(ita->text()) << "</title>\n<link>file://" << Qt::escape(notesFileName) << "</link>\n";
		out << "<description>" << Qt::escape(ita->toolTip()) << "</description>\n<pubDate>" << Qt::escape(QDateTime::currentDateTime().toString(Qt::TextDate)) << "</pubDate>\n<guid></guid>\n</item>\n\n";
	    }
	}
	out << "</channel>\n</rss>\n";
    }
}

void PTray::writeSettings()
{
     QSettings settings("Programmez", "QuickDesktopNotes");
     settings.setValue("pos", pos());
     settings.setValue("size", size());
     settings.setValue("notesFileName",notesFileName);
     settings.setValue("iconFileName",iconFileName);
}

void PTray::readSettings()
{
    QSettings settings("Programmez", "QuickDesktopNotes");
    QPoint pos = settings.value("pos", QPoint(200, 200)).toPoint();
    QSize size = settings.value("size", QSize(800, 600)).toSize();
    notesFileName = settings.value("notesFileName","desktop_notes.rss").toString();
    iconFileName = settings.value("iconFileName",":/images/burn.png").toString();
    setWindowIcon(QIcon(iconFileName));
    resize(size);
    move(pos);
}


void PTray::setVisible(bool visible)
{
    minimizeAction->setEnabled(visible);
    maximizeAction->setEnabled(!isMaximized());
    restoreAction->setEnabled(isMaximized() || !visible);
    PNoteEditor::setVisible(visible);
}

void PTray::closeEvent(QCloseEvent *event)
{
    if (trayIcon->isVisible() && !warnOnClose) {
        QMessageBox::information(this, tr("Quick Desktop Notes"),
                                 tr("The program will keep running in the "
                                    "system tray. To terminate the program, "
                                    "choose <b>Quit</b> in the context menu "
                                    "of the system tray entry."));
        warnOnClose=true;
	hide();
        event->ignore();
    }
    else if(warnOnClose){
	hide();
        event->ignore();
    }
}


void PTray::iconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
    case QSystemTrayIcon::Trigger:
	if(isVisible())
	    setVisible(false);
	else
	    showNormal();
	break;
    case QSystemTrayIcon::DoubleClick:
// 	showMessage();
        break;
    case QSystemTrayIcon::MiddleClick:
        showMessage();
        break;
    default:
        ;
    }
}

void PTray::showConfigDialog(){
    setVisible(true);
    PConfigDialog *config = new PConfigDialog(this);
    config->setMinimumSize(config->size());
    config->setMaximumSize(config->size());
    connect(config, SIGNAL(noteRemoved(QAction *)),this, SLOT(removeActionFromHistory(QAction *)));
    connect(config, SIGNAL(settingsUpdated()),this, SLOT(on_settingsUpdated()));
    config->setNoteList(history);
    config->show();
}

void PTray::on_settingsUpdated(){
    readSettings();
    trayIcon->setIcon(QIcon(iconFileName));
}

void PTray::removeActionFromHistory(QAction *action){
    /** interdiction de supprimer la "current note action" **/
    if(action != currentNoteAction){
	history->removeAction(action);
	trayIconMenu->removeAction(action);
    }
}

void PTray::showMessage()
{
    trayIcon->showMessage(noteTitle(), noteText(), QSystemTrayIcon::Information,15 * 1000);
}

void PTray::messageClicked()
{
    showNormal();
}

void PTray::on_noteTitleChanged(const QString &str){
    QAction *ita = history->checkedAction();
    ita->setText(str);
}

void PTray::on_noteTextChanged(const QString &str){
    QAction *ita = history->checkedAction();
    ita->setToolTip(str);
}

void PTray::addNoteToHistory(const QString &title, const QString &body){
    QAction *oldNote = new QAction( title, this );
    oldNote->setToolTip( body );
    oldNote->setCheckable(true);
    oldNote->setChecked(false);
    connect( oldNote, SIGNAL(triggered()), this, SLOT(editNote()) );
    trayIconMenu->addAction(oldNote);
    history->addAction(oldNote);
}

void PTray::newNote(){
    setVisible(true);
    addNoteToHistory( noteTitle(), noteText() );
    setNoteTitle("");
    setNoteText("");
    writeNotes();
}

void PTray::editNote(){
    QAction *action = qobject_cast< QAction* >( sender() );
    if(action){
	setNoteTitle( action->text() );
	setNoteText( action->toolTip() );
    }
}

void PTray::quitTray(){
    writeNotes();
    writeSettings();
    qApp->quit();
}

void PTray::createActions()
{
    minimizeAction = new QAction(tr("Mi&nimize"), this);
    minimizeAction->setIcon(QIcon(":/images/minimize.svg"));
    connect(minimizeAction, SIGNAL(triggered()), this, SLOT(hide()));

    maximizeAction = new QAction(tr("Ma&ximize"), this);
    maximizeAction->setIcon(QIcon(":/images/maximize.svg"));
    connect(maximizeAction, SIGNAL(triggered()), this, SLOT(showMaximized()));

    restoreAction = new QAction(tr("&Restore"), this);
    connect(restoreAction, SIGNAL(triggered()), this, SLOT(showNormal()));

    quitAction = new QAction(tr("&Quit"), this);
    quitAction->setIcon(QIcon(":/images/exit.svg"));
    connect(quitAction, SIGNAL(triggered()), this, SLOT(quitTray()));
    
    newNoteAction = new QAction(tr("&New note"), this );
    newNoteAction->setIcon(QIcon(":/images/list_add.svg"));
    connect(newNoteAction , SIGNAL(triggered()), this, SLOT(newNote()));
    
    currentNoteAction = new QAction(this);
    connect(currentNoteAction , SIGNAL(triggered()), this, SLOT(editNote()));
    
    configAction = new QAction( tr("Configure"),this );
    connect(configAction, SIGNAL(triggered()), this, SLOT(showConfigDialog()));
    configAction->setIcon(QIcon(":/images/configure.svg"));
    
    history = new QActionGroup(this);
    history->addAction(currentNoteAction);
    currentNoteAction->setCheckable(true);
    currentNoteAction->setChecked(true);
}

void PTray::createTrayIcon()
{
    trayIconMenu = new QMenu(this);
    trayIconMenu->addAction(minimizeAction);
    trayIconMenu->addAction(maximizeAction);
    trayIconMenu->addAction(restoreAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(configAction);
    trayIconMenu->addAction(newNoteAction);
    trayIconMenu->addAction(currentNoteAction);
    trayIconMenu->addSeparator();
    trayIconMenu->addAction(quitAction);
    trayIcon = new QSystemTrayIcon(this);
    trayIconMenu->addSeparator();
    trayIcon->setContextMenu(trayIconMenu);
    trayIcon->setIcon(QIcon(iconFileName));
}
