using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Storage;

namespace XnaPanic
{
    public class GameScreen : IGameScreen
    {
        // Services
        private INumberTracerService _numberTracerService;
        private IScoreService _scoreService;

        // Private Data
        private Game _game;
        private Grid _grid;
        private Man _man;
        private Monster[] _monsters;

        private bool _gameOver;
        private int _initialMonsterCount;
        private int _currentMonsterCount;
        private int _level;
        private int _score;
        private int _bestScore;
        private int _bonus;
        private int _lives;
        private MultiTextureSprite _scoreLabel;
        private MultiTextureSprite _bestLabel;
        private MultiTextureSprite _levelLabel;
        private MultiTextureSprite _bonusLabel;
        private MultiTextureSprite _lifeLabel;

        private const int BonusPeriod = 40;

        private int[,] MonstersDefinitions = {
            { 3, 0, 0 },
            { 5, 0, 0 },
            { 7, 0, 0 },
            { 2, 1, 0 },
            { 4, 1, 0 },
            { 6, 1, 0 },
            { 1, 1, 1 },
            { 3, 1, 1 },
            { 5, 1, 1 }
        };
        private int[] StartX = { 12, 48, 24, 60, 36, 24, 48 };
        private int[] StartY = { 48, 36, 60, 48, 48, 36, 60 };

        #region Properties
        public int InitialMonsterCount
        {
            get { return _initialMonsterCount; }
            set { _initialMonsterCount = value; }
        }

        public int CurrentMonsterCount
        {
            get { return _currentMonsterCount; }
        }

        public Man Man
        {
            get
            {
                return _man;
            }
        }
        #endregion

        #region Construction
        public GameScreen(Game game)
        {
            _game = game;
            _numberTracerService = (INumberTracerService)game.Services.GetService(typeof(INumberTracerService));
            _scoreService = (IScoreService)game.Services.GetService(typeof(IScoreService));

            _grid = new Grid(game);
            _man = new Man(this, _grid, game);

        }
        #endregion

        #region IGameScreen implementation
        public void LoadGraphicsContent(ContentManager content, int width, int height)
        {
            _levelLabel = new MultiTextureSprite(SpriteHorizontalAlignment.Left, SpriteVerticalAlignment.Top, content, @"Sprites\Game", "Level");
            _levelLabel.X = 10;
            _levelLabel.Y = 5;
            _scoreLabel = new MultiTextureSprite(SpriteHorizontalAlignment.Left, SpriteVerticalAlignment.Top, content, @"Sprites\Game", "Score");
            _scoreLabel.Y = 5;
            _bestLabel = new MultiTextureSprite(SpriteHorizontalAlignment.Left, SpriteVerticalAlignment.Top, content, @"Sprites\Game", "Best");
            _bestLabel.Y = 5;
            _bonusLabel = new MultiTextureSprite(SpriteHorizontalAlignment.Left, SpriteVerticalAlignment.Top, content, @"Sprites\Game", "Bonus");
            _bonusLabel.Y = 5;
            _lifeLabel = new MultiTextureSprite(SpriteHorizontalAlignment.Left, SpriteVerticalAlignment.Top, content, @"Sprites\Game", "Life");
            _lifeLabel.Y = 5;
            _man.LoadGraphicsContent(content, width, height);
            _grid.LoadGraphicsContent(content, width, height);
            Monster.LoadGraphicsContent(content, width, height);
        }



        // Pause implementation
        bool manualPause = false;
        bool switchPause = true;
        bool timePause = false;
        DateTime pauseUntil;

        public GameState Update(GamePadState gamepad, KeyboardState keyboard, GameTime gameTime)
        {
            // Handle Pause key
            if (gamepad.Buttons.Back == ButtonState.Pressed)
            {
                if (switchPause)
                {
                    manualPause = !manualPause;
                    switchPause = false;
                }
            }
            else
            {
                switchPause = true;
            }
            if (manualPause) return GameState.Play;

            // Handle timed pause
            if (timePause && (DateTime.Now < pauseUntil)) return GameState.Play;
            timePause = false;

            if (_bonus > 0)
            {
                _bonus--;
            }
            if (_gameOver) return GameState.Menu;

            // Check collisions with monsters
            bool deadMan = false;
            for (int i = 0; i < _initialMonsterCount; i++)
            {
                if (_monsters[i] != null)
                {
                    if (_man.Collide(_monsters[i]))
                    {
                        _man.Die();
                        deadMan = true;
                    }
                    for (int j = i + 1; j < _initialMonsterCount; j++)
                    {
                        if ((_monsters[j] != null) && _monsters[i].Collide(_monsters[j]))
                        {
                            // if one monster is falling, then both die
                            if (_monsters[i].Falling || _monsters[j].Falling)
                            {
                                // TODO : kill only "non falling" monster
                                _monsters[i].Die();
                                _monsters[j].Die();
                            }
                            else
                            {
                                _monsters[i].ChangeDir = true;
                                _monsters[j].ChangeDir = true;
                            }
                        }
                    }
                }
            }

            // Let the GameComponents update
            _man.Update(gamepad, keyboard, gameTime);
            _grid.Update(gamepad, keyboard, gameTime);
            foreach (Monster monster in _monsters)
            {
                if (monster != null)
                {
                    monster.Update(gamepad, keyboard, gameTime);
                }
            }

            if (deadMan)
            {
                pauseUntil = DateTime.Now + new TimeSpan(0, 0, 1);
                timePause = true;
            }
            return GameState.Play;
        }

        public void Draw(SpriteBatch spriteBatch)
        {
            _grid.Draw(spriteBatch);
            _man.Draw(spriteBatch);
            foreach (Monster monster in _monsters)
            {
                if (monster != null)
                {
                    monster.Draw(spriteBatch);
                }
            }

            // Now draw the current information
            _levelLabel.Draw(spriteBatch);
            _numberTracerService.Draw(spriteBatch, _level + 1, _levelLabel.X + _levelLabel.Width + 5, 5, 2);

            if (_scoreLabel.X == 0)
            {
                _scoreLabel.X = _levelLabel.X + _levelLabel.Width + 2 * _numberTracerService.Width + 15;
            }
            _scoreLabel.Draw(spriteBatch);
            _numberTracerService.Draw(spriteBatch, _score, _scoreLabel.X + _scoreLabel.Width + 5, 5, 5);

            if (_bestLabel.X == 0)
            {
                _bestLabel.X = _scoreLabel.X + _scoreLabel.Width + 5 * _numberTracerService.Width + 15;
            }
            _bestLabel.Draw(spriteBatch);
            _numberTracerService.Draw(spriteBatch, _bestScore, _bestLabel.X + _bestLabel.Width + 5, 5, 5);

            if (_bonusLabel.X == 0)
            {
                _bonusLabel.X = _bestLabel.X + _bestLabel.Width + 5 * _numberTracerService.Width + 15;
            }
            _bonusLabel.Draw(spriteBatch);
            _numberTracerService.Draw(spriteBatch, _bonus, _bonusLabel.X + _bonusLabel.Width + 5, 5, 4);

            _lifeLabel.X = _bonusLabel.X + _bonusLabel.Width + 4 * _numberTracerService.Width + 15;
            for (int i = 0; i < _lives; i++)
            {
                _lifeLabel.Draw(spriteBatch);
                _lifeLabel.X += _lifeLabel.Width;
            }
        }

        #endregion

        public void NewGame()
        {
            _bestScore = _scoreService.Best;
            _gameOver = false;
            _level = 0;
            _lives = 3;
            StartLevel(false);
        }

        public void StartLevel(bool nextLevel)
        {
            _bonus = 2000;

            if (nextLevel)
            {
                _level++;
            }
            _grid.StartLevel();
            _man.StartLevel();
            // Create new monsters according to level definition
            int levelIndex = (_level % MonstersDefinitions.Length);
            _initialMonsterCount = MonstersDefinitions[_level, 0] + MonstersDefinitions[_level, 1] + MonstersDefinitions[_level, 2];
            _currentMonsterCount = _initialMonsterCount;
            _monsters = new Monster[_initialMonsterCount];
            int monsterLevel = 0;
            int monsterCount = 0;
            for (int i = 0; i < _initialMonsterCount; i++)
            {
                Monster newMonster = new Monster(monsterLevel + 1, this, _grid, _game);
                if (++monsterCount == MonstersDefinitions[_level, monsterLevel])
                {
                    monsterLevel++;
                    monsterCount = 0;
                }
                newMonster.X = StartX[i];
                newMonster.Y = StartY[i];
                newMonster.Index = i;
                _monsters[i] = newMonster;
            }
        }

        public void KillMonster(Monster monster, int height, int level)
        {
            bool newLevel = true;
            for (int i = 0; i < _initialMonsterCount; i++)
            {
                if (_monsters[i] == monster)
                {
                    // TODO : increment score, taking into account height
                    _score += 50 * height * (height + 1) * level;
                    _currentMonsterCount--;
                    _monsters[i] = null;
                    for (int j = 0; j < _initialMonsterCount; j++)
                    {
                        if (_monsters[j] != null)
                        {
                            newLevel = false;
                            break;
                        }
                    }
                    break;
                }
            }
            if (newLevel)
            {
                if (_bonus > 0)
                {
                    _score += _bonus;
                }
                StartLevel(true);
            }
        }

        public void KillMan()
        {
            if (--_lives == 0)
            {
                _gameOver = true;
                _scoreService.GameScore(_score);
            }
            else
            {
                StartLevel(false);
            }
        }

    } // class
} // namespace