using System;
using System.Collections.Generic;
using System.Diagnostics;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Storage;
using Microsoft.Xna.Framework.Content;

namespace XnaPanic
{
    public enum ManPosture { RunningLeft, RunningRight, Climbing, DiggingLeft, DiggingRight, Falling, Dead };


    public class Man
    {
        #region Private data
        private GameScreen _gameScreen;
        private MultiTextureSprite[] _sprites;
        private MultiTextureSprite _currentSprite;
        private ManPosture _posture;
        private Grid _grid;
        private ISoundService _soundService;

        int _x;
        int _y;
        int _speedX;
        int _speedY;
        int _digDir;
        bool _digging;
        bool _falling;
        bool _dead;
        ManPosture _oldPosture;
        private BoundingBox _bb;

        #endregion

        #region Properties
        public ManPosture Posture
        {
            get { return _posture; }
            set {
                if (_posture != value)
                {
                    _posture = value;
                    if (_sprites != null)
                    {
                        _currentSprite = _sprites[(int)_posture];
                        if (_currentSprite != null)
                        {
                            _currentSprite.TextureIndex = 0;
                            _currentSprite.X = _grid.GetX(_x, _currentSprite.HorizontalAlignment);
                            _currentSprite.Y = _grid.GetY(_y);
                        }
                    }
                }
            }
        }
        public MultiTextureSprite CurrentSprite
        {
            get { return _currentSprite; }
        }
        public int X
        {
            get { return _x; }
            set { _x = value; }
        }
        public int Y
        {
            get { return _y; }
            set { _y = value; }
        }
        #endregion

        #region Construction
        public Man(GameScreen gameScreen, Grid grid, Game game)
        {
            _gameScreen = gameScreen;
            _grid = grid;
            _bb = new BoundingBox();
            _soundService = (ISoundService)game.Services.GetService(typeof(ISoundService));
        }
        #endregion

        public void LoadGraphicsContent(ContentManager content, int width, int height)
        {
#if XBOX
            _sprites = new MultiTextureSprite[7];
#else
            _sprites = new MultiTextureSprite[Enum.GetValues(typeof(ManPosture)).Length];
#endif
            _sprites[(int)ManPosture.RunningLeft] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManLeft1", "ManLeft2");
            _sprites[(int)ManPosture.RunningRight] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManRight1", "ManRight2");
            _sprites[(int)ManPosture.Climbing] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManClimbing1", "ManClimbing2");
            _sprites[(int)ManPosture.DiggingRight] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManDigRight1", "ManDigRight2");
            _sprites[(int)ManPosture.DiggingRight].OffsetX = 12;
            _sprites[(int)ManPosture.DiggingLeft] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManDigLeft1", "ManDigLeft2");
            _sprites[(int)ManPosture.DiggingLeft].OffsetX = -12;
            _sprites[(int)ManPosture.Falling] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManFalling");
            _sprites[(int)ManPosture.Falling].OffsetX = 6;
            _sprites[(int)ManPosture.Dead] = new MultiTextureSprite(SpriteHorizontalAlignment.Center, SpriteVerticalAlignment.Bottom, content, @"Sprites\Man", "ManKilled");
        }


        public void Update(GamePadState gamepad, KeyboardState keyboard, GameTime gameTime)
        {
            if (_falling || _dead) {
                UpdatePos();
            }
            else
            {
                _speedX = 0;
                _speedY = 0;
                if ((gamepad.ThumbSticks.Left.X <= -0.8) || keyboard.IsKeyDown(Keys.Left))
                {
                    if ((_posture != ManPosture.RunningLeft))
                    {
                        if (_grid.CanGo(_x, _y, -1, 0, false))
                        {
                            Posture = ManPosture.RunningLeft;
                        }
                        else
                        {
                            // if just one level above or below ground, force it
                            int newY = 3 * ((_y + 1) / 3);
                            if (_grid.CanGo(_x, newY, -1, 0, false))
                            {
                                _y = newY;
                                Posture = ManPosture.RunningLeft;
                            }
                        }
                    }
                    else
                    {
                        _speedX = -1;
                        UpdatePos();
                    }
                }
                else if ((gamepad.ThumbSticks.Left.X >= 0.8) || keyboard.IsKeyDown(Keys.Right))
                {
                    if (_posture != ManPosture.RunningRight)
                    {
                        if (_grid.CanGo(_x, _y, 1, 0, false))
                        {
                            Posture = ManPosture.RunningRight;
                        }
                        else
                        {
                            // if just one level above or below ground, force it
                            int newY = 3 * ((_y + 1) / 3);
                            if (_grid.CanGo(_x, newY, 1, 0, false))
                            {
                                _y = newY;
                                Posture = ManPosture.RunningLeft;
                            }
                        }
                    }
                    else
                    {
                        _speedX = 1;
                        UpdatePos();
                    }
                }
                else if ((gamepad.ThumbSticks.Left.Y >= 0.8) || keyboard.IsKeyDown(Keys.Up))
                {
                    if (_posture != ManPosture.Climbing)
                    {
                        if (_grid.CanGo(_x, _y, 0, 1, false))
                        {
                            Posture = ManPosture.Climbing;
                        }
                        else // if just one step right or left to the scale, force it
                            if (_grid.CanGo(_x + 1, _y, 0, 1, false))
                            {
                                _x++;
                                Posture = ManPosture.Climbing;
                            }
                            else if (_grid.CanGo(_x - 1, _y, 0, 1, false))
                            {
                                _x--;
                                Posture = ManPosture.Climbing;
                            }
                    }
                    else
                    {
                        _speedY = 1;
                        UpdatePos();
                    }
                }
                else if ((gamepad.ThumbSticks.Left.Y <= -0.8) || keyboard.IsKeyDown(Keys.Down))
                {
                    if (_posture != ManPosture.Climbing)
                    {
                        if (_grid.CanGo(_x, _y, 0, -1, false))
                        {
                            Posture = ManPosture.Climbing;
                        }
                        else // if just one step right or left to the scale, force it
                            if (_grid.CanGo(_x + 1, _y, 0, -1, false))
                            {
                                _x++;
                                Posture = ManPosture.Climbing;
                            }
                            else if (_grid.CanGo(_x - 1, _y, 0, -1, false))
                            {
                                _x--;
                                Posture = ManPosture.Climbing;
                            }
                    }
                    else
                    {
                        _speedY = -1;
                        UpdatePos();
                    }
                }
                else if ((gamepad.Buttons.A == ButtonState.Pressed) || keyboard.IsKeyDown(Keys.LeftControl))
                {
                    switch (_posture)
                    {
                        case ManPosture.RunningRight:
                        case ManPosture.DiggingRight:
                            if (_grid.CanDig(_x, _y, 1))
                            {
                                Posture = ManPosture.DiggingRight;
                                _digDir = 1;
                                _digging = true;
                            }
                            break;
                        case ManPosture.RunningLeft:
                        case ManPosture.DiggingLeft:
                            if (_grid.CanDig(_x, _y, -1))
                            {
                                Posture = ManPosture.DiggingLeft;
                                _digDir = -1;
                                _digging = true;
                            }
                            break;
                    }
                    if ((Posture == ManPosture.DiggingRight) || (Posture == ManPosture.DiggingLeft))
                    {
                        UpdatePos();
                    }
                }
                else if ((gamepad.Buttons.B == ButtonState.Pressed) || keyboard.IsKeyDown(Keys.LeftShift))
                {
                    switch (_posture)
                    {
                        case ManPosture.RunningRight:
                        case ManPosture.DiggingRight:
                            if (_grid.CanFill(_x, _y, 1))
                            {
                                Posture = ManPosture.DiggingRight;
                                _digDir = 1;
                                _digging = false;
                            }
                            break;
                        case ManPosture.RunningLeft:
                        case ManPosture.DiggingLeft:
                            if (_grid.CanFill(_x, _y, -1))
                            {
                                Posture = ManPosture.DiggingLeft;
                                _digDir = -1;
                                _digging = false;
                            }
                            break;
                    }
                    if ((Posture == ManPosture.DiggingRight) || (Posture == ManPosture.DiggingLeft))
                    {
                        UpdatePos();
                    }
                }
            }
        }

        static int _digTempo = 0;
        Cue _cue;
        protected void UpdatePos() {
            bool nextSprite = false;

            if (_dead)
            {
                if (Posture == ManPosture.Dead)
                {
                    // Second loop
                    _gameScreen.KillMan();
                }
                else
                {
                    Posture = ManPosture.Dead;
                }
            }
            else
            {
                if ((_speedX != 0) || (_speedY != 0))
                {
                    // Two separate if just for debug...
                    if (_falling)
                    {
                        _y += _speedY;
                        if (_grid.EndFall(_x, _y))
                        {
                            _speedX = 0;
                            _speedY = 0;
                            Posture = _oldPosture;
                            _falling = false;
                            if ((_cue != null) && _cue.IsPlaying)
                            {
                                _cue.Stop(AudioStopOptions.Immediate);
                            }
                        }
                    }
                    else
                    {
                        if (_grid.CanGo(_x, _y, _speedX, _speedY, false))
                        {
                            _x += _speedX;
                            _y += _speedY;
                            if (_grid.IsInHole(_x, _y) && _grid.IsInHole(_x-1, _y) && _grid.IsInHole(_x+1, _y))
                            {
                                _falling = true;
                                _oldPosture = Posture;
                                Posture = ManPosture.Falling;
                                _x = _grid.GetHolePos(_x, _y);
                                _currentSprite.TextureIndex = 0;
                                _speedX = 0;
                                _speedY = -1;
                                _cue = _soundService.PlaySound(Sound.Fall);
                            }
                            else
                            {
                                nextSprite = true;
                                _soundService.PlaySound(Sound.Step);
                            }
                        }
                    }
                }
                else if ((_posture == ManPosture.DiggingRight) || (_posture == ManPosture.DiggingLeft))
                {
                    if (++_digTempo >= 5)
                    {
                        _digTempo = 0;
                        _currentSprite.IncrementTexture();
                        if (_currentSprite.TextureIndex == 0)
                        {
                            _grid.Dig(_x, _y, _digDir, _digging);
                        }
                    }
                }
            }
            if (_currentSprite != null) {
                _currentSprite.X = _grid.GetX(_x, _currentSprite.HorizontalAlignment);
                _currentSprite.Y = _grid.GetY(_y);
                if (nextSprite) {
                    _currentSprite.IncrementTexture();
                }
            }
            _bb.Min = new Vector3((float)_currentSprite.X, (float)_currentSprite.Y, 0);
            _bb.Max = new Vector3((float)(_currentSprite.X + _currentSprite.Width), (float)(_currentSprite.Y + _currentSprite.Height), 0);

        }

        public void Draw(SpriteBatch spriteBatch)
        {
            _currentSprite.Draw(spriteBatch);
        }

        public void StartLevel() {
            Posture = ManPosture.RunningRight;
            _x = 5;
            _y = 0;
            _speedX = 0;
            _speedY = 0;
            _digDir = 1;
            _falling = false;
            _dead = false;
            _digging = false;
            UpdatePos();
        }
        public bool Collide(Monster monster)
        {
            return (!monster.InHole) && _bb.Intersects(monster.BoundingBox);
        }

        public void Die()
        {
            this._dead = true;
        }
    }
}